#ifndef __DCL_ARRAY_T_H__
#define __DCL_ARRAY_T_H__		20110304

#ifndef __DCL_CONFIG_H__
	#include <dcl/Config.h>
#endif

#if !__DCL_DEBUG && !defined(__DCL_INCLUDED_STDLIB_H)
	#include <stdlib.h>		// malloc, free, realloc
	#define __DCL_INCLUDED_STDLIB_H
#endif
#ifndef __DCL_INCLUDED_STRING_H
	#include <string.h>		// memset, memmove
	#define __DCL_INCLUDED_STRING_H
#endif

#ifndef __DCL_OBJECT_H__
#include <dcl/Object.h>
#endif

__DCL_BEGIN_NAMESPACE

#if __DCL_HAVE_ALLOC_DEBUG
	#undef __DCL_ALLOC_LEVEL
	#define __DCL_ALLOC_LEVEL	__DCL_ALLOC_INTERNAL
	#undef new
	#define new __DCL_DEBUG_NEW
#endif

#if __DCL_HAVE_THIS_FILE__
	static const char_t __szArrayT_h__[] = __T("dcl/ArrayT.h");
	#undef __THIS_FILE__
	#define __THIS_FILE__ __szArrayT_h__
#endif

/**
 * ELEMENT에 대한 배열을 표현한다.
 */

template<typename ELEMENT>
class Array : public Object
{
	DECLARE_CLASSINFO(Array)
public:
	typedef const ELEMENT*	ConstIterator;
	typedef ELEMENT*			Iterator;

public:
	virtual ~Array();
	// ~Array();
	Array(size_t _size = 0);
	Array(const Array<ELEMENT>& _src);
	const Array<ELEMENT>& operator=(const Array<ELEMENT>& _src);

	ConstIterator begin() const;
	ConstIterator end() const;
	Iterator begin();
	Iterator end();

	void clear();
	void shrink();
	void resize(size_t _size);

	// @return is inserted element Iterator
	Iterator insert(Iterator _pos, const ELEMENT& _element);
	Array<ELEMENT>& insert(size_t _index, const ELEMENT& _element);
	Array<ELEMENT>& add(const ELEMENT& _element);
	Array<ELEMENT>& add(const Array<ELEMENT>& _src);

	Iterator find(const ELEMENT& _element);

	// @return is next Iterator
	Iterator erase(Iterator _pos);					
	Iterator erase(Iterator _first, Iterator _last);
	Array<ELEMENT>& erase(size_t _index);
	Array<ELEMENT>& erase(size_t _index, size_t _size);

	size_t index(Iterator _pos) const;
	size_t size() const;
	bool isEmpty() const;

	ELEMENT& operator[](size_t _index);
	ELEMENT operator[](size_t _index) const;
	ELEMENT* data() const;

	size_t size(ConstIterator _first, ConstIterator _last) const;
	size_t size(Iterator _first, Iterator _last) const;

protected:
	ELEMENT*	__pData;

	struct Buffer
	{
		size_t	__size;
		size_t	__maxSize;
		ELEMENT* data() { return (ELEMENT*) (this + 1); }
	} ;
	
	Buffer* __buf() const { return (Buffer*) __pData - 1; }
	size_t& __size() const { return __buf()->__size; }
	size_t& __maxSize() const { return __buf()->__maxSize; }

	void constructElements(ELEMENT* _pElements, size_t _size);
	void destructElements(ELEMENT* _pElements, size_t _size);
};

//////////////// Inline ////////////////////

template<typename ELEMENT>
inline
typename Array<ELEMENT>::ConstIterator
Array<ELEMENT>::begin() const
{
	return (const ELEMENT*) __pData;
}

template<typename ELEMENT>
inline
typename Array<ELEMENT>::ConstIterator
Array<ELEMENT>::end() const
{
	return (const ELEMENT*) __pData + size();
}

template<typename ELEMENT>
inline
typename Array<ELEMENT>::Iterator
Array<ELEMENT>::begin()
{
	return __pData;
}

template<typename ELEMENT>
inline
typename Array<ELEMENT>::Iterator
Array<ELEMENT>::end()
{
	return __pData + size();
}

template<typename ELEMENT>
inline
Array<ELEMENT>&
Array<ELEMENT>::add(const ELEMENT& _element)
{
	return insert(size(), _element);
}

template<typename ELEMENT>
inline
typename Array<ELEMENT>::Iterator
Array<ELEMENT>::erase(Iterator _pos)
{
	__DCL_ASSERT_PARAM(begin() <= _pos);
	__DCL_ASSERT_PARAM(_pos < end());
#if 0
	erase(_pos - begin(), 1);
	return _pos;
#endif
	size_t index = _pos - begin();
	erase(index, 1);
	return __pData + index;
}

template<typename ELEMENT>
inline
typename Array<ELEMENT>::Iterator
Array<ELEMENT>::erase(Iterator _first, Iterator _last)
{
	__DCL_ASSERT_PARAM(begin() <= _first);
	__DCL_ASSERT_PARAM(_last <= end());
	erase(_first - begin(), _last - _first);
	return _first;
}

template<typename ELEMENT>
inline
Array<ELEMENT>&
Array<ELEMENT>::erase(size_t _index)
{
	__DCL_ASSERT_PARAM(_index < size());
	return erase(_index, 1);
}

template<typename ELEMENT>	
inline
size_t
Array<ELEMENT>::index(Iterator _pos) const
{
	__DCL_ASSERT_PARAM(__pData <= _pos);
	return _pos - __pData;
}

template<typename ELEMENT>
inline
size_t
Array<ELEMENT>::size() const
{
	return __buf()->__size;;
}

template<typename ELEMENT>
inline
bool
Array<ELEMENT>::isEmpty() const
{
	return size() == 0;
}

template<typename ELEMENT>
inline
ELEMENT&
Array<ELEMENT>::operator[] (size_t _index)
{
	__DCL_ASSERT_PARAM(_index < size());
	return __pData[_index];
}

template<typename ELEMENT>
inline
ELEMENT
Array<ELEMENT>::operator[] (size_t _index) const
{
	__DCL_ASSERT_PARAM(_index < size());
	return __pData[_index];
}

template<typename ELEMENT>
inline
ELEMENT*
Array<ELEMENT>::data() const
{
	return __pData;
}

template<typename ELEMENT>	
inline
size_t
Array<ELEMENT>::size(ConstIterator _first, ConstIterator _last) const
{
	__DCL_ASSERT_PARAM(_first <= _last);
	return _last - _first;
}

template<typename ELEMENT>	
inline
size_t
Array<ELEMENT>::size(Iterator _first, Iterator _last) const
{
	__DCL_ASSERT_PARAM(_first <= _last);
	return _last - _first;
}

//////////////////////// Implement //////////////////////////

// IMPLEMENT_CLASSINFO(Array<ELEMENT>, Object)
#ifdef __DCL_NO_RTTI
template<typename ELEMENT>
const Object::ClassInfo Array<ELEMENT>::m_classInfo =
{
	__DCL_NAMESPACE_STRING __S(Array),
	sizeof(class Array),
	&Object::m_classInfo
};

template<typename ELEMENT>
const Object::ClassInfo* Array<ELEMENT>::classInfo() const
{
	return CLASSINFO(class_name);
}
#else
template<typename ELEMENT>
const std::type_info& Array<ELEMENT>::typeInfo() const
{
	return typeid(Array);
}
#endif

template<typename ELEMENT>
inline
void
Array<ELEMENT>::constructElements(ELEMENT* _pElements, size_t _size)
{
	for(; _size; _size--, _pElements++)
	{
#if __DCL_HAVE_ALLOC_DEBUG
#undef new
#endif

		new((void*)_pElements) ELEMENT;

#if __DCL_HAVE_ALLOC_DEBUG
#define new __DCL_DEBUG_NEW
#endif
	}
}

template<typename ELEMENT>
inline
void
Array<ELEMENT>::destructElements(ELEMENT* _pElements, size_t _size)
{
	for( ; _size; _size--, _pElements++)
	{
		_pElements->~ELEMENT();
	}
}

template<typename ELEMENT>
Array<ELEMENT>::Array(size_t _size)
{
	__pData = NULL;
	resize(_size);
}

template<typename ELEMENT>
Array<ELEMENT>::Array(const Array<ELEMENT>& _src)
{
	__pData = NULL;
	*this = _src;
}

template<typename ELEMENT>
Array<ELEMENT>::~Array()
{
	clear();
	free(__buf());
}

template<typename ELEMENT>
const
Array<ELEMENT>&
Array<ELEMENT>::operator=(const Array<ELEMENT>& _src)
{
	if (&_src != this)
	{
		resize(_src.size());
		for(size_t i = 0; i < _src.size(); i++)
			__pData[i] = _src.__pData[i];
	}
	return *this;
}

template<typename ELEMENT>
void
Array<ELEMENT>::clear()
{
	if (size() > 0)
	{
		destructElements(__pData, size());
		__size() = 0;
	}
}

template<typename ELEMENT>
void
Array<ELEMENT>::shrink()
{
	if (size() <__maxSize())
	{
		Buffer* buf = (Buffer*) realloc(__buf(), sizeof(Buffer) + size() * sizeof(ELEMENT));
		// FIXME!
		// if (buf == NULL)
		//		throw new BadAllocException();
		__DCL_ASSERT(buf != NULL);

		buf->__maxSize = buf->__size;
		__pData = buf->data();
	}
}

template<typename ELEMENT>
void
Array<ELEMENT>::resize(size_t _size)
{
	if (__pData != NULL)
	{
		if (size() == _size)
			return;

		if (size() > _size)
		{
			destructElements(__pData + _size, size() - _size);
			__size() = _size;
			return;
		}
	}

	if (__pData == NULL || __maxSize() < _size)
	{
		Buffer* buf = NULL;
		if (__pData == NULL)
		{
			buf = (Buffer*) malloc(sizeof(Buffer) + _size * sizeof(ELEMENT));
			// FIXME!
			// if (buf == NULL)
			//		throw new BadAllocException();
			__DCL_ASSERT(buf != NULL);

			buf->__size = 0;
		}
		else
		{
			buf = (Buffer*) realloc(__buf(), sizeof(Buffer) + _size * sizeof(ELEMENT));
			// FIXME!
			// if (buf == NULL)
			//		throw new BadAllocException();
			__DCL_ASSERT(buf != NULL);
		}

		buf->__maxSize = _size;
		__pData = buf->data();
		memset((void*)(__pData + buf->__size), 0, (_size - buf->__size) * sizeof(ELEMENT));
	}

	constructElements(__pData + size(), _size - size());
	__size() = _size;
}

template<typename ELEMENT>
typename Array<ELEMENT>::Iterator
Array<ELEMENT>::insert(Iterator _pos, const ELEMENT& _element)
{
	__DCL_ASSERT_PARAM(begin() <= _pos);
	__DCL_ASSERT_PARAM(_pos <= end());

	size_t index = _pos - __pData;
	insert(index, _element);
	return __pData + index + 1;
}

template<typename ELEMENT>
Array<ELEMENT>&
Array<ELEMENT>::insert(size_t _index, const ELEMENT& _element)
{
	__DCL_ASSERT_HANDLE(__pData != NULL);
	__DCL_ASSERT_PARAM(_index <= size());

	size_t newSize = size() + 1;
	if (__maxSize() < newSize)
	{
		Buffer* buf =  (Buffer*) realloc(__buf(), sizeof(Buffer) + newSize * sizeof(ELEMENT)); 
		// FIXME!
		// if (buf == NULL)
		//		throw new BadAllocException();
		__DCL_ASSERT(buf != NULL);

		buf->__maxSize = newSize;
		__pData = buf->data();

		if (_index < buf->__size)
			memmove((void*)(__pData + _index + 1), (const void*)(__pData + _index),
				sizeof(ELEMENT) * (buf->__size - _index));
	}

	constructElements(__pData + _index, 1);
	__size() = newSize;
	__pData[_index] = (ELEMENT) _element;
	return *this;
}

template<typename ELEMENT>
Array<ELEMENT>&
Array<ELEMENT>::add(const Array<ELEMENT>& _src)
{
	if (_src.size() > 0)
	{
		size_t newSize = size() + _src.size();
		if (__maxSize() < newSize)
		{
			Buffer* buf =  (Buffer*) realloc(__buf(), sizeof(Buffer) + newSize * sizeof(ELEMENT));
			// FIXME!
			// if (buf == NULL)
			//		throw new BadAllocException();
			__DCL_ASSERT(buf != NULL);

			__pData = buf->data();
		}

		constructElements(__pData + size(), _src.size());
		for(size_t i = 0; i < _src.size(); i++)
			__pData[size() + i] = _src.__pData[i];

		__maxSize() = __size() = newSize;
	}
	return *this;
}

template<typename ELEMENT>
typename Array<ELEMENT>::Iterator
Array<ELEMENT>::find(const ELEMENT& _element)
{
	Iterator it = begin();
	for ( ; it != end(); it++)
	{
		if (*it == _element)
			break;
	}
	return it;
}

template<typename ELEMENT>
Array<ELEMENT>&
Array<ELEMENT>::erase(size_t _index, size_t _size)
{
	__DCL_ASSERT_PARAM(_index + _size <= size());
	if (_size > 0)
	{
		destructElements(__pData + _index, _size);
		if (_index + _size < size())
			memmove((void*)(__pData + _index), (const void*)(__pData + _index + _size),
				 (size() -  (_index  + _size)) * sizeof(ELEMENT));
		__size() -= _size;
	}
	return *this;
}


#if __DCL_HAVE_THIS_FILE__
	#undef __THIS_FILE__
	#define __THIS_FILE__	__T(__FILE__)
#endif

#if __DCL_HAVE_ALLOC_DEBUG
	#undef __DCL_ALLOC_LEVEL
	#define __DCL_ALLOC_LEVEL	__DCL_ALLOC_USER
	#undef new
	#define new __DCL_DEBUG_NEW
#endif

__DCL_END_NAMESPACE

#endif			// __DCL_ARRAY_T_H__
