#ifndef __DCL_BUFFERED_INPUT_STREAM_H__
#define __DCL_BUFFERED_INPUT_STREAM_H__		20110130

#ifndef __DCL_INPUT_STREAM_H__
#include <dcl/InputStream.h>
#endif
#ifndef __DCL_STRING_H__
#include <dcl/String.h>
#endif

__DCL_BEGIN_NAMESPACE

/**
 * 바이트 스트림으로 부터 줄(line)단위 읽기를 가능하게 한다.
 * 각각의 라인은 CR, LF 또는 CRLF로 종결될 수 있다.
 *
 * @author	Daejung Kim
 * @since		DCL Version 3.0
 *
 * @see		InputStream
 * @see		BufferedReader
 */
class DCLCAPI BufferedInputStream : public InputStream
{
	DECLARE_CLASSINFO(BufferedInputStream)
public:
	virtual String toString() const;

	/**
	 * 객체를 구성한다.
	 * _input는 {@link #close()}에서 close하지 않으며, 생명주기동안 유효해야 한다.
	 * <p>다음은 사용의 예를 보여주며, 스택을 빠져 나가면,
	 * 각각은 _input, input 순서로는 닫히며 파괴된다.</p>
	 * <pre>
	 *      FileInputStream _input(filename);
	 *      BufferedInputStream input(_input);
	 *      String s;
	 *      if (input.readLine(s))
	 *          output << s << endl;
	 *	</pre>
	 * @param _input
	 *		A InputStream
	 * @param _bufSize
	 *		_input를 위한 버퍼로, _input의 읽기 단위가 된다. 반드시 0보다 커야 한다.
	 */
	BufferedInputStream(InputStream& __noclose__ _input, size_t _bufSize = 1024);

	/**
	 * 객체를 구성한다. _pInput는 {@link #close()}에서 close하고 destroy한다.
	 * <p>다음은 사용의 예를 보여주며, 스택을 빠져 나가면 input는 자동으로 닫힌다.</p>
	 * <pre>
	 *      BufferedInputStream input(new FileInputStream(filename));
	 *      String s;
	 *      if (input.readLine(s))
	 *          output << s << endl;
	 * </pre>
	 * @param _pInput
	 *		A InputStream
	 * @param _bufSize
	 *		_input를 위한 버퍼로, _input의 읽기 단위가 된다. 반드시 0보다 커야 한다.
	 */
	BufferedInputStream(InputStream* __destroy__ _pInput, size_t _bufSize = 1024)
			__DCL_THROWS1(IOException*);

	/**
	 * 객체의 파괴를 수행한다.
	 * close하지 않았으면 {@link #close()}하며 예외가 발생하면 무시한다.
	 */
	virtual ~BufferedInputStream();

	/**
	 * 버퍼를 지우고 객체를 닫는다.
	 * {@link #BufferedInputStream(InputStream*)}로 열린 입력 객체를 close하고 destroy한다.
	 */
	virtual void close()
			__DCL_THROWS1(IOException*);

	/**
	 * 버퍼의 데이터를 포함하여 텍스트를 읽는다.
	 * <p>이 메소드는 다음의 상태에 리턴한다.</p>
	 * <ul>
	 *    <li>_n개의 요구된 문자를 모두 읽은 경우</li>
	 *    <li>EOF인 경우</li>
	 *    <li>블록되었음을 감지한 경우</li>
	 * </ul>
	 *
	 * @param _buf
	 *		버퍼
	 * @param _n
	 *		_buf의 길이, _buf가 가리키는 버퍼의 char의 개수
	 * @return
	 *		읽혀진 문자의 개수이다. 0이면 EOF이다.
	 */
	virtual size_t read(void* _buf, size_t _n)
			__DCL_THROWS1(IOException*);

	/**
	 * 한줄을 읽는다.
	 * <p>한 줄이 버퍼의 크기를 초과하면 CR, LF 또는 CRLF가 발견될 때까지
	 * 입력 스트림으로 부터 read를 반복한다.
	 * 입력 스트림이 블록될 수 있는 경우, 한 줄의 길이가 버퍼의 크기를 넘지 않으면
	 * 1번을 초과하여 블록되지 않는다.</p>
	 *
	 * @param _buf
	 *		리턴될 라인 버퍼이다. CR, LF 또는 CRLF를 포함하지 않느다.
	 * @return
	 *		false이면 EOF이다.
	*/
	virtual bool readLine(ByteString& _buf)
			__DCL_THROWS1(IOException*);

private:
	InputStream*		__input;
	bool				__closeDestroy;

	char*				__buf;
	size_t				__bufSize;
	const char*		__begin;
	const char*		__end;
	bool				__endLine;
};

__DCL_END_NAMESPACE

#endif		// __DCL_BUFFERED_INPUT_STREAM_H__
