#ifndef __DCL_BUFFERED_READER_H__
#define __DCL_BUFFERED_READER_H__		20110130

#ifndef __DCL_READER_H__
#include <dcl/Reader.h>
#endif
#ifndef __DCL_STRING_H__
#include <dcl/String.h>
#endif

__DCL_BEGIN_NAMESPACE

/**
 * 텍스트 스트림으로 부터 줄(line)단위 읽기를 가능하게 한다.
 * 각각의 라인은 CR, LF 또는 CRLF로 종결될 수 있다.
 *
 * @author	Daejung Kim
 * @since		DCL Version 3.0
 *
 * @see		InputStreamReader
 * @see		FileReader
 */
class DCLCAPI BufferedReader : public Reader
{
	DECLARE_CLASSINFO(BufferedReader)
public:
	virtual String toString() const;

	/**
	 * 객체를 구성한다.
	 * _reader는 {@link #close()}에서 close하지 않으며, 생명주기동안 유효해야 한다.
	 * <p>다음은 사용의 예를 보여주며, 스택을 빠져 나가면,
	 * 각각은 reader, _reader, input 순서로는 닫히며 파괴된다.</p>
	 * <pre>
	 *      FileInputStream input(filename);
	 *      InputStreamReader _reader(input);
	 *      BufferedReader reader(_reader);
	 *      String s;
	 *      if (reader.readLine(s))
	 *          writer << s << endl;
	 *	</pre>
	 * @param _reader
	 *		A Reader
	 * @param _bufSize
	 *		_reader를 위한 버퍼로, _reader의 읽기 단위가 된다. 반드시 0보다 커야 한다.
	 */
	BufferedReader(Reader& __noclose__ _reader, size_t _bufSize = 1024);

	/**
	 * 객체를 구성한다. _pReader는 {@link #close()}에서 close하고 destroy한다.
	 * <p>다음은 사용의 예를 보여주며, 스택을 빠져 나가면 reader는 자동으로 닫힌다.</p>
	 * <pre>
	 *      BufferedReader reader(new InputStreamReader(new FileInputStream(filename)));
	 *      String s;
	 *      if (reader.readLine(s))
	 *          writer << s << endl;
	 * </pre>
	 * @param _pReader
	 *		A Reader
	 * @param _bufSize
	 *		_reader를 위한 버퍼로, _reader의 읽기 단위가 된다. 반드시 0보다 커야 한다.
	 */
	BufferedReader(Reader* __destroy__ _pReader, size_t _bufSize = 1024)
			__DCL_THROWS1(IOException*);

	/**
	 * 객체의 파괴를 수행한다.
	 * close하지 않았으면 {@link #close()}하며 예외가 발생하면 무시한다.
	 */
	virtual ~BufferedReader();

	/**
	 * 버퍼를 지우고 객체를 닫는다.
	 * {@link #BufferedReader(Reader*)}로 열린 입력 객체를 close하고 destroy한다.
	 */
	virtual void close()
			__DCL_THROWS1(IOException*);

	/**
	 * 버퍼의 데이터를 포함하여 텍스트를 읽는다.
	 * <p>이 메소드는 다음의 상태에 리턴한다.</p>
	 * <ul>
	 *    <li>_n개의 요구된 문자를 모두 읽은 경우</li>
	 *    <li>EOF인 경우</li>
	 *    <li>블록되었음을 감지한 경우</li>
	 * </ul>
	 *
	 * @param _buf
	 *		버퍼
	 * @param _n
	 *		_buf의 길이, _buf가 가리키는 버퍼의 wchar_t의 개수
	 * @return
	 *		읽혀진 문자의 개수이다. 0이면 EOF이다.
	 */
	virtual size_t read(wchar_t* _buf, size_t _n)
			__DCL_THROWS1(IOException*);

	/**
	 * 한줄을 읽는다.
	 * <p>한 줄이 버퍼의 크기를 초과하면 CR, LF 또는 CRLF가 발견될 때까지
	 * 입력 스트림으로 부터 read를 반복한다.
	 * 입력 스트림이 블록될 수 있는 경우, 한 줄의 길이가 버퍼의 크기를 넘지 않으면
	 * 1번을 초과하여 블록되지 않는다.</p>
	 *
	 * @param _buf
	 *		리턴될 라인 버퍼이다. CR, LF 또는 CRLF를 포함하지 않느다.
	 * @return
	 *		false이면 EOF이다.
	*/
	virtual bool readLine(String& _buf)
			__DCL_THROWS1(IOException*);

private:
	Reader*		__reader;
	bool			__closeDestroy;

	wchar_t*			__buf;
	size_t				__bufSize;
	const wchar_t*	__begin;
	const wchar_t*	__end;
	bool				__endLine;
};

__DCL_END_NAMESPACE

#endif		// __DCL_BUFFERED_READER_H__
