#ifndef __DCL_BYTES_OUTPUT_STREAM_H__
#define __DCL_BYTES_OUTPUT_STREAM_H__		20110221

#ifndef __DCL_OUTPUT_STRAEM_H__
#include <dcl/OutputStream.h>
#endif
#ifndef __DCL_STRING_H__
#include <dcl/String.h>
#endif

__DCL_BEGIN_NAMESPACE

/**
 * 바이트 스트림을 버퍼링하여 ByteString을 만든다.
 *
 * @author	Daejung Kim
 * @since		DCL Version 3.0
 *
 * @see		ByteBuffer
 * @see		ByteString
 * @see		OutputSteam
 */
class DCLCAPI BytesOutputStream : public OutputStream
{
	DECLARE_CLASSINFO(BytesOutputStream)
public:
	/**
	 * 버퍼링된 데이터로 ByteString 객체를 구성하여 반환다.
	 */
	ByteString toByteString() const;

	/**
	 * 주어진 값으로 객체를 구성한다.
	 * <p>
	 * {@link #write(constvoid*,size_t) write(const void* _buf, size_t _n)}이
	 * 호출되기 전에는 내부버퍼를 생성하지 않는다.
	 * </p>
	 * @param _initial
	 *		내부버퍼가 생성될 때 char 문자의 개수
	 * @param _grow
	 *		내부버퍼의 증가 크기
	 */
	BytesOutputStream(size_t _initial = 4096);

	/**
	 * 객체의 파괴를 수행한다.
	 * close하지 않았으면 {@link #close()}하며 예외가 발생하면 무시한다.
	 */
	virtual ~BytesOutputStream();

	/**
	 * 버퍼를 삭제하고 스트림을 닫는다.
	 */
	virtual void close()
			__DCL_THROWS1(IOException*);

	/**
	 * _n바이트의 데이터를 스트림에 write한다.
	 * @param _buf
	 *		버퍼
	 * @param _n
	 *		버퍼의 크기가 _n바이트 
	 * @return
	 *		OutputStream&
	 */
	virtual OutputStream& write(const void* _buf, size_t _n)
			__DCL_THROWS1(IOException*);

	/**
	 * 주어진 OutputStream으로 내부버퍼의 텍스트를 쓴다. 내부버퍼는 그대로 유지한다.
	 *
	 * @param _output
	 *		A OutputStream
	 * @return
	 *		_output에 쓴 바이트의 개수
	 */
	virtual size_t writeTo(OutputStream& _output) const
			__DCL_THROWS1(IOException*);

	/**
	 * 형식화된 문자열을 출력한다.
	 * @param _format
	 *		포멧 문자열. %s는 %hs로 해석된다.
	 * @param _arglist
	 *		아귀먼트 리스트
	 * @return
	 *		출력된 총 바이트수를 반환한다.
	 */
	virtual int vprintf(const char* _format, va_list _arglist)
		__DCL_THROWS1(IOException*);

	/**
	 * 주어진 OutputStream로 내부버퍼의 텍스트를 내보내고, 내부버퍼를 reset한다.
	 *
	 * @param _writer
	 *		A Writer
	 * @return
	 *		_writer에 쓴 char 문자의 개수
	 */
	virtual size_t flushTo(OutputStream& _output)
			__DCL_THROWS1(IOException*);

	/**
	 * 내부버퍼를 초기화 한다.
	 */
	virtual void reset();

	/**
	 * 내부버퍼의 시작 위치를 반환하다.
	 * 만약, 단 한번도 {@link #write(constchar*,size_t) write(const void* _buf, size_t _n)}이 호출되지 않았으면,
	 * NULL을 반환할 것이다.
	 */
	const void* data() const;

	/**
	 * 내부버퍼에 저장되어 있는 바이트의 개수 반환한다. 
	 */
	size_t size() const;

protected:
	ByteBuffer*		__buf;
	size_t			__capacity;
};

inline const void* BytesOutputStream::data() const
{
	return __buf != NULL ? __buf->data() : NULL;
}

inline size_t BytesOutputStream::size() const
{
	return __buf != NULL ? __buf->__dataLength : 0;
}

__DCL_END_NAMESPACE

#endif		// __DCL_BYTES_OUTPUT_STREAM_H__
