#ifndef __DCL_DATE_TIME_H__
#define __DCL_DATE_TIME_H__		20050323

#ifndef __DCL_CONFIG_H__
#include <dcl/Config.h>
#endif
#ifndef __DCL_STRING_H__
#include <dcl/String.h>
#endif
#ifndef __DCL_EXCEPTION_H__
#include <dcl/Exception.h>
#endif

__DCL_BEGIN_NAMESPACE

class Date;
class Time;
class Interval;
class DateTime;

#define DATETIME_FORMAT_BUFFER_SIZE		256
// valid range -9999-01-01 ~ 9999-12-31
// internal range -599999-01-01 ~ 600000-12-31
class DCLCAPI Date
{
public:
	Date();
	Date(const Date& _src);
	Date(int _year, int _month, int _day);

	void assign(int _year, int _month, int _day);

	// _days == 0 : NULL
	// _days > 0 : 0001-01-01 ~
	// _days < 0 : ~ -0001-12-31
	Date(long _days);
	void assign(long _days);
	long days() const;

//	long jdays() const { return __jday; }

	const Date& operator ++ ();
	const Date operator ++ (int);
	const Date& operator -- ();
	const Date operator -- (int);

	const Date& operator = (const Date& _src);
	const Date& operator += (long _days);
	const Date& operator -= (long _days);
	
	void decode(int& _year, int& _month, int& _day) const;

	int year() const;
	int month() const;
	int day() const;

	int dayOfWeek() const;
	int dayOfYear() const;
	int daysInMonth() const;
	int daysInYear() const;

	// yyyy-mm-dd
	String toString() const;

	// strftime
	static const wchar_t* FORMAT_STRING;	// ISO 8601 format %Y-%m-%d

	String toStringF(const wchar_t* _format = NULL) const;


	// 0001-01-01 <= date <= 9999-12-31
	static bool isValid(int y, int m, int d);

	static bool isLeapYear(int _year);

protected:
	long	__jday;

	static long convertGregorianToJulian(
				int _year, int _month, int _day
				);

	static void convertJulianToGregorian(
				long uJulianDays,
				int& _year, int& _month, int& _day
				);
};

DCLCAPI Date operator + (const Date& _date, long _days);
DCLCAPI Date operator + (long _days, const Date& _date);	
DCLCAPI Date operator - (const Date& _date, long _days);

// return days
DCLCAPI long operator - (const Date& _date1, const Date& _date2);

DCLCAPI bool operator == (const Date& _date1, const Date& _date2);
DCLCAPI bool operator != (const Date& _date1, const Date& _date2);
DCLCAPI bool operator >= (const Date& _date1, const Date& _date2);
DCLCAPI bool operator <= (const Date& _date1, const Date& _date2);
DCLCAPI bool operator > (const Date& _date1, const Date& _date2);
DCLCAPI bool operator < (const Date& _date1, const Date& _date2);

class DCLCAPI Time
{
public:
	Time();
	Time(const Time& _src);
	Time(int _hour, int _min, int _sec, int _millisecond = 0);
	Time(long _milliseconds);

	void assign(int _hour, int _min, int _sec, int _millisecond = 0);
	void assign(long _milliseconds);

	const Time& operator = (const Time& _src);
	const Time& operator += (long _milliseconds);
	const Time& operator -= (long _milliseconds);

	void decode(int& _hour, int& _min, int& _sec, int& _millisecond) const;

	int hour() const;		// 0 ~ 23
	int minute() const;		// 0 ~ 59
	int second() const;		// 0 ~ 59
	int msecond() const;	// 0 ~ 999

	// HH:MM:SS.MS 00:00:00.000
	String toString() const;

	// strftime
	static const wchar_t* FORMAT_STRING;

	String toStringF(const wchar_t* _format = NULL);

	long totalMilliSeconds() const;	// 0 ~ (86400000 - 1)

	static bool isValid(int _hour, int _min, int _sec, int _millisecond);


protected:
	unsigned long	__milliseconds;
};

DCLCAPI Time operator + (const Time& _time, long _milliseconds);
DCLCAPI Time operator + (long _milliseconds, const Time& _time);
DCLCAPI Time operator - (const Time& _time, long _milliseconds);

// return milliseconds
DCLCAPI long operator - (const Time& _time1, const Time& _time2);

DCLCAPI bool operator == (const Time& _time1, const Time& _time2);
DCLCAPI bool operator != (const Time& _time1, const Time& _time2);
DCLCAPI bool operator >= (const Time& _time1, const Time& _time2);
DCLCAPI bool operator <= (const Time& _time1, const Time& _time2);
DCLCAPI bool operator > (const Time& _time1, const Time& _time2);
DCLCAPI bool operator < (const Time& _time1, const Time& _time2);

class DCLCAPI Interval
{
public:
	Interval();
	Interval(const Interval& _src);
	Interval(int _days, int _milliseconds);
	Interval(int _days,
			int _hours, int _minutes, int _seconds,
			int _milliseconds = 0
			);
	Interval(int64_t _milliseconds);

	void assign(int _days, int _milliseconds);
	void assign(int _days,
			int _hours, int _minutes, int _seconds,
			int _milliseconds = 0
			);
	void assign(int64_t _milliseconds);

	const Interval& operator = (const Interval& _src);
	const Interval& operator += (const Interval& _iv);
	const Interval& operator -= (const Interval& _iv);

	void decode(long& _days,
			int& _hours, int& _minutes, int& _seconds,
			int& _milliseconds
			) const;
	void decode(long& _days, long& _milliseconds) const;

	long days() const;
	int hour() const;		// -23 ~ 23
	int minute() const;		// -59 ~ 59
	int second() const;		// -59 ~ 59
	int msecond() const;	// -999 ~ 999

	int64_t totalHours() const;
	int64_t totalMinutes() const;
	int64_t totalSeconds() const;
	int64_t totalMilliSeconds() const;

	// [-]000 days [-]00:00:00.000
	String toString() const;

protected:
	int64_t	__milliseconds;
};

DCLCAPI Interval operator + (const Interval& _iv1, const Interval& _iv2);
DCLCAPI Interval operator - (const Interval& _iv1, const Interval& _iv2);

DCLCAPI bool operator == (const Interval& _iv1, const Interval& _iv2);
DCLCAPI bool operator != (const Interval& _iv1, const Interval& _iv2);
DCLCAPI bool operator >= (const Interval& _iv1, const Interval& _iv2);
DCLCAPI bool operator <= (const Interval& _iv1, const Interval& _iv2);
DCLCAPI bool operator > (const Interval& _iv1, const Interval& _iv2);
DCLCAPI bool operator < (const Interval& _iv1, const Interval& _iv2);

class DCLCAPI DateTime
{
public:
	DateTime() { }
	DateTime(const DateTime& _src);
	DateTime(time_t _timer);
	DateTime(const Date& _date, const Time& _time);
	DateTime(int _year, int _month, int _day,
			int _hour, int _min, int _sec, int _millisecond = 0);

	void assign(time_t _timer);
	void assign(const Date& _date, const Time& _time);
	void assign(int _year, int _month, int _day,
			int _hour, int _min, int _sec, int _millisecond = 0);

	const DateTime& operator = (const DateTime& _src);
	const DateTime& operator += (const Interval& _iv);
	const DateTime& operator -= (const Interval& _iv);

	// Date::toString() + Time::toString()
	String toString() const;

	// stftime
	static const wchar_t* FORMAT_STRING;

	String toStringF(const wchar_t* _format = NULL) const;

	Date& date();
	const Date& date() const;

	Time& time();
	const Time& time() const;

	static int compare(const DateTime& _dt1, const DateTime& _dt2);

	static int64_t getCurrentTimeMillis();
	static DateTime getCurrentUTCTime();
	static DateTime getCurrentLocalTime();
	
/*
	static time_t parse(const char* pszDateTimeString)
				__DCL_THROWS1(ParseException*);

	// strptime
	static DateTime parseF(
			const wchar_t* pszDateTimeString,
			const wchar_t* _format
			) __DCL_THROWS1(ParseException*);
*/

protected:
	Date	__date;
	Time	__time;

};

DCLCAPI DateTime operator + (const DateTime& _dt, const Interval& _iv);
DCLCAPI DateTime operator + (const Interval& _iv, const DateTime& _dt);
DCLCAPI DateTime operator - (const DateTime& _dt, const Interval& _iv);
DCLCAPI Interval operator - (const DateTime& _dt1, const DateTime& _dt2);

DCLCAPI bool operator == (const DateTime& _dt1, const DateTime& _dt2);
DCLCAPI bool operator != (const DateTime& _dt1, const DateTime& _dt2);
DCLCAPI bool operator >= (const DateTime& _dt1, const DateTime& _dt2);
DCLCAPI bool operator <= (const DateTime& _dt1, const DateTime& _dt2);
DCLCAPI bool operator > (const DateTime& _dt1, const DateTime& _dt2);
DCLCAPI bool operator < (const DateTime& _dt1, const DateTime& _dt2);


#include "DateTime.inl"

__DCL_END_NAMESPACE

#endif	// __DCL_DATE_TIME_H__

