#ifndef __DCL_DIR_H__
#define __DCL_DIR_H__

#ifndef __DCL_CONFIG_H__
#include <dcl/Config.h>
#endif

#if __DCL_WINDOWS
	#ifndef _WINDOWS_
		#error "Required windows.h, See dcl/_windows.h"
	#endif
#else
	#include <dirent.h>
#endif

#ifndef __DCL_OBJECT_H__
#include <dcl/Object.h>
#endif
#ifndef __DCL_STRING_H__
#include <dcl/String.h>
#endif
#ifndef __DCL_EXCEPTION_H__
#include <dcl/Exception.h>
#endif

__DCL_BEGIN_NAMESPACE

class CharsetConvertException;

/**
 * 디렉터리 목록에 읽는다.
 * UNIX에서는 dirent.h로 구현되었고, Windows는 FileFirstFile을 사용한다.
 *
 * @author	Daejung Kim
 * @since	DCL Version 1.0
*/
class DCLCAPI Dir : public Object
{
	DECLARE_CLASSINFO(Dir)
public:
#if __DCL_WINDOWS
	typedef HANDLE HandleType;
#else
	typedef DIR* HandleType;
#endif

	/**
	 * 디렉터리 엔트리를 추상화 한다.
	 * UNIX는 struct dirent, Windows는 WIN32_FIND_DATA를 상속하며,
	 * 이 클래스에서 직접 지원하지 않는 속성들은 해당 운영체제 메뉴얼을 참고한다.
	 */
	class DCLCAPI Entry
#if __DCL_WINDOWS
		: public WIN32_FIND_DATAW
#else
		: public dirent
#endif
	{
	public:
		/**
		 * 디렉터리 여부를 판단한다.
		 */
		bool isDir() const;

		/**
		 * 심볼릭 링크 여부를 판단한다.
		 * Windows에서 디렉터리는 심볼릭 링크일 수 있다.
		 */
		bool isLink() const;

		/**
		 * 객체를 초기화 한다.
		 */
		Entry();

		/**
		 * 엔트리의 파일이름을 리턴한다.
		 * UNIX에서 파일이름 변환에서 예외가 발생할 수 있다.
		 *
		 * @throw CharsetConvertException*
		 * @see File
		 */

		String name() const
				__DCL_THROWS1(CharsetConvertException*);

		/**
		 * 엔트리의 파일이름을 포함하여 요약 문자열을 리턴한다.
		 */
		String toString() const;
	};

	/**
	 * 객체의 파괴를 수행한다.
	 * close하지 않았으면 {@link #close()}하며 예외가 발생하면 무시한다.
	 */
	virtual ~Dir();

	/**
	 *	객체를 구성한다.
	 * _path가 없거나, 디렉터리가 아니면 예외가 발생한다.
	 */
	Dir(const String& _path)
				__DCL_THROWS1(IOException*);

	/**
	 * 객체를 닫는다.
	 */
	void close()
				__DCL_THROWS1(IOException*);

	/**
	 * 읽는 위치를 처음으로 되돌린다.
	 */
	void rewind()
				__DCL_THROWS1(IOException*);

	/**
	 * 하나의 엔트리를 읽는다.
	 * <p>다음은 간략한 사용을 보여준다</p>
	 * <pre>
	 *	String path = __T(".");
	 *	Dir dir(path);
	 *	Dir::Entry entry;
	 *	while (dir.read(entry))
	 *	    writer << entry.toString() << endl;
	 * </pre>
	 *
	 * @return false이면 EOF이다.
	 */
	bool read(Entry& _buf) __DCL_THROWS1(IOException*);

	/**
	 * 디렉터리의 경로를 얻는다.
	 * 각각의 구분은 '/'이고, 문자열의 끝에는 항상 '/'으로 종결된다.
	 */
	const String& path() const { return __path; }

protected:
	HandleType		__handle;
	String			__path;
};

#if __DCL_WINDOWS
inline bool Dir::Entry::isDir() const
{
	return (this->dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0;
}

inline bool Dir::Entry::isLink() const
{
	return (this->dwFileAttributes & FILE_ATTRIBUTE_REPARSE_POINT) != 0;
}
#else
inline bool Dir::Entry::isDir() const
{
	return this->d_type == DT_DIR;
}

inline bool Dir::Entry::isLink() const
{
	return this->d_type == DT_LNK;
}
#endif

__DCL_END_NAMESPACE

#endif	// __DCL_DIR_H__
