#ifndef __DCL_EXCEPTION_H__
#define __DCL_EXCEPTION_H__

#ifndef __DCL_CONFIG_H__
#include <dcl/Config.h>
#endif

#if __DCL_WINDOWS
	#ifndef _WINERROR_
		#include <WinError.h>
	#endif
#else
	#include <errno.h>
#endif

#ifndef __DCL_OBJECT_H__
#include <dcl/Object.h>
#endif
#ifndef __DCL_STRING_H__
#include <dcl/String.h>
#endif

__DCL_BEGIN_NAMESPACE

class DCLCAPI Exception : public Object
{
	DECLARE_CLASSINFO(Exception);
protected:
	Exception*	__pCause;

public:
	Exception(Exception* _cause = NULL);

	// 예외가 발생한 원인 예외 객체
	const Exception* cause() const;

	// 예외와 관련한 메시지 문자열
	virtual String toString() const;	

	// 원인을 포함한 메시지
	String toStringAll() const;

	// no delete e, use e->destroy()
	virtual void destroy();	

protected:
	virtual ~Exception() {}		// cannot delete e, use e->destroy()
};

inline const Exception* Exception::cause() const
{
	return __pCause;
}

class DCLCAPI OutOfMemoryException : public Exception
{
	DECLARE_CLASSINFO(OutOfMemoryException)
public:
	OutOfMemoryException();
};

class DCLCAPI SysError : public Exception
{
	DECLARE_CLASSINFO(SysError)
protected:
	uint32_t	__errorNo;

#define ERROR_VALUE_MASK		0x0fffffff
#define ERRNO_ERROR_MASK		0x00000000
#define WINAPI_ERROR_MASK		0x10000000
#define DCL_ERROR_MASK			0x20000000
#define WINAPI_ERROR(n)			WINAPI_ERROR_MASK | n
#define DCL_ERROR(n)			DCL_ERROR_MASK | n

public:
	SysError(Exception* _cause = NULL);
	SysError(uint32_t _errorNo);
	uint32_t errorNo() const { return __errorNo; }
	virtual String toString() const;
};

#if __DCL_WINDOWS
	#define __EHANDLE		ERROR_INVALID_HANDLE
	#define __EPARAM		ERROR_INVALID_PARAMETER
	#define __ENOMEM		ERROR_NOT_ENOUGH_MEMORY
#else
	#define __EHANDLE		EINVAL
	#define __EPARAM		EENVAL
	#define __ENOMEM		ENOMEM
#endif

class DCLCAPI IOException : public SysError
{
	DECLARE_CLASSINFO(IOException)
public:
	enum
	{
		NOMEM		= __ENOMEM
	};
protected:
	String		__message;

public:
	IOException(const String& _name, Exception* _cause);
	IOException(const String& _name, uint32_t _errorNo);
	IOException(const String& _name, const String& _message);
	virtual String toString() const;
};

class DCLCAPI InvalidIndexException : public Exception
{
	DECLARE_CLASSINFO(InvalidIndexException)
protected:
	String	__message;
public:
	InvalidIndexException(const String& _key);
	InvalidIndexException(ssize_t _minValid, ssize_t _maxValid, size_t _errorValue);
	virtual String toString() const;
};

class DCLCAPI ParseException : public Exception
{
	DECLARE_CLASSINFO(ParseException)
protected:
	String	__message;
	size_t	__sourceOffset;

public:
	ParseException(const String& _message);
	ParseException(const String& _message, size_t _sourceOffset);
	virtual String toString() const;
};

class DCLCAPI NumericConvertException : public ParseException
{
	DECLARE_CLASSINFO(NumericConvertException)
public:
	enum Error
	{
		NoError = 0,
		InvalidNumberString,
		Overflow,
		Underflow
	};

	// eInvalidNumberString
	NumericConvertException(
			const String& _number,
			int _radix,
			size_t _sourceOffset
			);

	// eOverflow or eUnderflow
	NumericConvertException(
			Error _error,
			const String& _number,
			int _radix
			);

	virtual String toString() const;

	Error errorCode() const { return __error; }

private :
	Error __error;
	int		__radix;
};

class DCLCAPI GenerialException : public Exception
{
	DECLARE_CLASSINFO(GenerialException)
protected:
	String	__message;
public:
	GenerialException(Exception* _cause);
	GenerialException(const wchar_t* _message);
	virtual String toString() const;
};

class DCLCAPI AssertError : public Exception
{
	DECLARE_CLASSINFO(AssertError)
public:
	String		__expr;
	String		__message;
	String		__filename;		// __FILE__
	int			__line;			// __LINE__
public:
	AssertError(
			const char_t*		_filename, 
			unsigned int		_line,
			const char_t*		_expr,
			const char_t*		_message
			);

	virtual String toString() const;

	static String format(
			const char_t*		_filename, 
			unsigned int		_line,
			const char_t*		_expr,
			const char_t*		_message
			);
};

/*
#define __DCL_CHECK_THROW(_expr, _constructor) \
	if (_expr) throw (new _constructor)

	__DCL_CHECK_THROW(_format == NULL,
		IOException(toString(), IOException::EPARAM));
*/

/*
class DCLCAPI SocketException : public SysCallException
{
	DECLARE_CLASSINFO(SocketException)
protected:
	String __message;
public:
	SocketException(const char* _message);
	virtual String toString() const;
};
*/
__DCL_END_NAMESPACE

#endif	// __DCL_EXCEPTION_H__

