#ifndef __DCL_FILE_H__
#define __DCL_FILE_H__		20041127

#ifndef __DCL_CONFIG_H__
#include <dcl/Config.h>
#endif

#if __DCL_WINDOWS
	#ifndef _WINDOWS_
		#error "Required windows.h, See dcl/_windows.h"
	#endif
#endif

#ifndef __DCL_OBJECT_H__
#include <dcl/Object.h>
#endif
#ifndef __DCL_STRING_H__
#include <dcl/String.h>
#endif
#ifndef __DCL_CHARSET_H__
#include <dcl/Charset.h>
#endif
#ifndef __DCL_DATE_TIME_H__
#include <dcl/DateTime.h>
#endif
#ifndef __DCL_EXCEPTION_H__
#include <dcl/Exception.h>
#endif

__DCL_BEGIN_NAMESPACE

/**
 * 운영체제 파일을 추상화 한다.
 * <p>이 클래스의 목적은 서로다른 운영체제에서 파일 접근을 위한 표준 접근방법을 제공하는 것으로,
 * 기본 모델은 UNIX의 것을 따른다.</p>
 */
class DCLCAPI File : public Object
{
	DECLARE_CLASSINFO(File)
public:
#if __DCL_WINDOWS
	typedef HANDLE HandleType;
	#define STDIN_HANDLE		GetStdHandle(STD_INPUT_HANDLE)
	#define STDOUT_HANDLE		GetStdHandle(STD_OUTPUT_HANDLE)
	#define STDERR_HANDLE		GetStdHandle(STD_ERROR_HANDLE)
	typedef __int64 off_t;
#else
	typedef int HandleType;
	#ifdef STDIN_FILENO
		#define STDIN_HANDLE		STDIN_FILENO
		#define STDOUT_HANDLE		STDOUT_FILENO
		#define STDERR_HANDLE		STDERR_FILENO
	#else
		#define STDIN_HANDLE		0
		#define STDOUT_HANDLE		1
		#define STDERR_HANDLE		2
	#endif
	typedef long long off_t;
#endif

	enum OpenFlags
	{
		READONLY		= 0x0001,		// O_RDONLY
		WRITEONLY		= 0x0002,		// O_WRONLY
		READWRITE		= 0x0004,		// O_RDWR
		CREATE			= 0x0010,		// O_CREAT
		EXCLUSIVE		= 0x0020,		// O_EXCL
		NOCTTY			= 0x0040,		// O_NOCTTY
		APPEND			= 0x0100,		// O_APPEND
		TRUNCATE		= 0x0200,		// O_TRUNC
		NONBLOCK		= 0x1000,		// O_NONBLOCK, FILE_FLAG_OVERLAPPED
		SYNC			= 0x2000		// O_SYNC
	};

protected:
	File();

	/**
	 * 열려진 파일핸들을 사용하여 객체를 구성한다.
	 * <p>이것은 {@link #openTempFile(const String&,const String&,int)} 사용한다.
	 * 
	 */
	File(HandleType _handle, const String& _path);

public:
	virtual String toString() const;

	/**
	 * 객체의 파괴를 수행한다.
	 * close하지 않았으면 {@link #close()}하며 예외가 발생하면 무시한다.
	 */
	virtual ~File();

	/**
	 * 주어진 경로의 파일을 열고 객체를 구성한다.
	 * <p>파일의 종류는 일반파일(Regular File), 시리얼 포트(COM#N, /dev/ttyS#N, /dev/ttyUSBS#N), 
	 * Windows에서 콘솔(COM), 명명된 파이프(Named Pipe)이다.</p>
	 *
	 * @param _path
	 * 		Windows에서 COM#N이 사용되면 큐 버퍼는 2048로 설정된다.
	 *
	 * @param _oflags
	 *		Windows CON은 READONLY, WRITEONLY만 사용된다.
	 * 		Windows에서 NONBLOCK을 사용하면 FILE_FLAG_OVERRLAPPED가 적용되고,
	 *		read, write를 위한 이벤트를 생성한다.
	 *
	 * @param _mode
	 *		Windows에서는 무시되고, UNIX에서 umask가 적용된다.
	 */
	File(const String& _path, int _oflags = READONLY, int _mode = 0666)
			__DCL_THROWS1(IOException*);

	/**
	 * 이미 열린 파일은 닫고, 파일을 새로 연다.
	 *
	 * @see {@link #File(constString&,int,int) File(const String& _path, int _oflags, int _mode)}
	 */
	void open(const String& _path, int _oflags = READONLY, int _mode = 0666)
			__DCL_THROWS1(IOException*);

	/**
	 * 열려진 파일핸들을 사용하여 객체를 구성한다.
	 *
	 * @see File::OpenFlags
	 *
	 * @param _handle
	 *		UNIX에서 open, pipe등의 시스템콜 리턴값, Windows에서는 CreateFile, CreatePipe 등의 리턴값이다.
	 * @param _ohints
	 *		Windows에서 _handle의 생성 플래그에 FILE_FLAG_OVERLAPPED가 사용되었거나, 
	 *		ioctlsock FIONBIO의 사용으로 NONBLOCK 으로 설정된 핸들에 대하여 사용된다.
	 *		READONLY, WRITEONLY, READWRITE와 NONBLOCK이 조합된다.
	 *		UNIX에서는 무시된다.
	 * @param _closeOnClose
	 *		{@link close()}에서 _handle을 닫을지 여부를 지정한다.
	 */
	File(HandleType _handle, int _ohints, bool _closeOnClose)
			__DCL_THROWS1(IOException*);

	/**
	 * 이미 열려진 파일핸들을 사용하여 객체를 구성한다.
	 *
	 * @see {@link File(HandleType,int,bool) File(HandleType _handle, int _ohints, bool _closeOnClose}
	 */
	void open(HandleType _handle, int _ohints, bool _closeOnClose)
			__DCL_THROWS1(IOException*);

	/**
	 * 장치와 동기화 한다.
	 * UNIX fsync, Windows FlushFileBuffers
	 */
	virtual void sync()
			__DCL_THROWS1(IOException*);

	/**
	 * 파일을 닫는다.
	 */
	virtual void close()
			__DCL_THROWS1(IOException*);

	/**
	 * Non-Block으로 읽혀질 수 있는 바이트의 수를 리턴한다.
	 * <p>UNIX에서 ioctl FIONREAD를 사용한다.
	 * WINDOWS에서는 일반파일, (Named) Pipe, 시리얼포트(COM#N), 콘솔 및 소켓에 대하여 구현되어 있고,
	 * 그 외에는 항상 0을 리턴한다.
	*/
	virtual size_t available() const
			__DCL_THROWS1(IOException*);

	/**
	 * 파일을 읽는다.
	 * 오픈 플래그에 NONBLOCK이 사용되면 0을 리턴할 수 있다.
	 * UNIX는 O_NONBLOCK이, Windows는 FILE_FLAG_OVERLAPPED가 사용된다.
	 * UNIX에서 리턴값이 0이면 errno가 EAGAIN 일 수 있다.
	 *
	 * @param _buf
	 *		버퍼
	 * @param _n
	 *		버퍼의 크기, 버퍼의 바이트 수
	 * @return
	 *		읽혀진 바이트의 수이다. 0이면 EOF를 의미한다.
	 */
	virtual size_t read(void* _buf, size_t _n)
			__DCL_THROWS1(IOException*);

	/**
	 * 파일에 데이터를 쓴다.
	 * 오픈 플래그에 NONBLOCK이 사용되면 오류 없이 0을 리턴할 수 있다.
	 * UNIX는 O_NONBLOCK이, Windows는 FILE_FLAG_OVERLAPPED가 사용된다.
	 * UNIX에서 리턴값이 0이면 errno가 EAGAIN 일 수 있다.
	 * 0을 리턴하는 것은 read, write를 호출한 스레드가 서로 달라서, 서로 겹쳐지기 때문으로 볼 수 있다.
	 * 0을 리턴하면 write하지 않았기 때문에 다시 시도해야 한다.
	 *
	 * @param _buf
	 *		버퍼
	 * @param _n
	 *		버퍼의 크기, 버퍼의 바이트 수 
	 * @return
	 *		쓰여진 바이트의 수이다. 0이면 다시 시도하라.
	 */
	virtual size_t write(const void* _buf, size_t _n)
			__DCL_THROWS1(IOException*);

	enum Whence
	{
		BEGIN,			// SEEK_SET
		CURRENT,		// SEEK_CUR
		END				// SEEK_END
	};

	/**
	 * 파일의 읽기/쓰기 변위(offset)를 재설정한다.
	 * @param _offset
	 *		변위
	 * @param _whence
	 *		BEGIN, CURRENT, END
	 * @return
	 *		파일의 시작을 기준으로 새로운 변위를 리턴한다.
	 */
	off_t seek(off_t _offset, int _whence)
			__DCL_THROWS1(IOException*);

	/**
	 * 파일의 크기를 구한다.
	 */
	off_t size() const
			__DCL_THROWS1(IOException*);

	/**
	 * 파일 최종 수정된 시간을 얻는다.
	 */
	DateTime mtime() const
			__DCL_THROWS1(IOException*);

	/*
	 * 파일의 최종 접근, 수정, 생성 시간을 얻는다.
	 */
	bool time(time_t* _atime, time_t* _mtime, time_t* _ctime) const;

	/**
	 * 파일의 핸들값을 리턴한다. (File::HandleType)-1 이면 닫혀진 상태이다.
	 */
	HandleType handle() const { return __handle; }

	/**
	 * 열려진 파일의 경로를 리턴한다.
	 */
	const String& path() const { return __path; }

protected:
	String			__path;
	HandleType		__handle;
	bool			__closeOnClose;
#if __DCL_WINDOWS
public:
	enum FileType
	{
		UNKNOWN,
		REGULAR,
		PIPE,
		CONSOLE,
		COMM,
		SOCKET
	};

	/**
	 * Windows에서 파일핸들의 종류를 리턴한다.
	 */
	FileType fileType() const { return __fileType; }
protected:
	FileType		__fileType;
	HANDLE			__readEvent;
	HANDLE			__writeEvent;
#endif

public:
	// "dir/prefixXXXXX" 이름의 파일을 생성하여 파일 기술자를 리턴
	static File* openTempFile(
					const String& _dirname,
					const String& _prefix,
					unsigned int _mode = 0666
					) __DCL_THROWS1(IOException*);
};

__DCL_END_NAMESPACE

#endif	// __DCL_FILE_H__
