#ifndef __DCL_FILES_H__
#define __DCL_FILES_H__     20241218

#ifndef __DCL_CONFIG_H__
#include <dcl/Config.h>
#endif
#ifndef __DCL_STRING_H__
#include <dcl/String.h>
#endif
#ifndef __DCL_CHARSET_H__
#include <dcl/Charset.h>
#endif
#ifndef __DCL_DATE_TIME_H__
#include <dcl/DateTime.h>
#endif
#ifndef __DCL_EXCEPTION_H__
#include <dcl/Exception.h>
#endif

__DCL_BEGIN_NAMESPACE

class InputStream;
class OutputStream;

class DCLCAPI Files
{
private:
	Files() {}

public:
	// static members
	// File2.cpp

	enum AccessType
	{
		READ_OK = 4,
		WRITE_OK = 2,
		EXECUTE_OK = 1,	// windows 무시
		EXISTS = 0,
	};

	/**
	프로세스의 접근권한을 검사
	path가 있고 성공하면 true, 그 외에는 모두 false
	@return == false 이면 errno 검사
	*/
	static bool access(const String& _path, int _mode);

	static void rename(const String& _oldpath, const String& _newpath)
		__DCL_THROWS1(IOException*);

	// unlink and rmdir
	static void remove(const String& _path)
		__DCL_THROWS1(IOException*);

	// path를 삭제
	static void unlink(const String& _path)
		__DCL_THROWS1(IOException*);

	// path 의 존재 유무 체크access(path, F_OK), 있으면 true
	// path가 심볼릭 링크이면 원본 체크
	static bool exists(const String& _path);

	// 파일의 디렉토리 여부를 판별한다.
	static bool isDirectory(const String& _path);

	/**
	 * 파일 최종 수정된 시간을 얻는다.
	 */
	static DateTime mtime(const String& _path)
		__DCL_THROWS1(IOException*);

	/*
	 * 파일의 최종 접근, 수정, 생성 시간을 얻는다.
	 */
	static bool time(const String& _path,
		time_t* _atime, time_t* _mtime, time_t* _ctime);

	// 파일이 없거나 접근 불능등의 에러이면 예외
	static uint64_t size(const String& _path)
		__DCL_THROWS1(IOException*);

	// 프로세스의 작업 디렉토리 변경
	static void chdir(const String& _path)
		__DCL_THROWS1(IOException*);

	// 프로세스의 현재 디렉토리(current working directory)
	static String getcwd() __DCL_THROWS1(IOException*);

	// 디렉토리 생성, windows에서 nMode 무시
	static void mkdir(const String& _path, int _mode = 0755)
		__DCL_THROWS1(IOException*);

	// 디렉토리 삭제
	static void rmdir(const String& _path)
		__DCL_THROWS1(IOException*);

	// path에서 파일명을 리턴, "dir/a" ==> "a"
	static String basename(const String& _path);

	// path에서 디렉토리명을 리턴, "dir/abc" ==> "dir/"
	// 리턴값문자열의 끝은 디렉토리 구분자('/', '\', ':')를 포함
	static String dirname(const String& _path);

	// path에서 디렉토리 구분자(default '/')를 리턴
	static wchar_t delimiter(const String& _path);

	// pszDirName에서 디렉토리 구분자를 찾아서 
	// pszDirName + 'delimiter char + pszBaseName 리턴
	static String filename(
		const String& _dirname,
		const String& _basename
	);

	// $TMP, $TEMP, /tmp 중 리턴, 셋 모두 실패하면 빈 String
	// 리턴값문자열의 끝은 디렉토리 구분자('/', '\')를 포함
	static String temppath();

	// 상대경로명을 절대경로명으로 리턴
	static String realpath(const String& _path)
		__DCL_THROWS1(IOException*);

	// "dir\dir\name" return "dir/dir/name"
	static String unixpath(const String& _path);

	// \, /, "C:, D:, ... return true
	static bool isAbsPath(const String& _path);

	static ByteString readBytes(InputStream& _input, size_t _n = (size_t)-1)
		__DCL_THROWS1(IOException*);

	static ByteString readBytes(const String& _filename, size_t _n = (size_t)-1)
		__DCL_THROWS1(IOException*);

	// UTF8, UTF16, UTF32, LC_CTYPE
	static String readText(const String& _filename)
		__DCL_THROWS1(IOException*);

	static String readText(const String& _filename, CharsetDecoder& _decoder)
		__DCL_THROWS1(IOException*);

	static size_t copy(InputStream& _input, OutputStream& _output)
		__DCL_THROWS1(IOException*);
};

__DCL_END_NAMESPACE

#endif  // __DCL_FILES_H__
