#ifndef __DCL_HTTP_COLLECTION_H__
#define __DCL_HTTP_COLLECTION_H__		20050526

#ifndef __DCL_OBJECT_H__
#include <dcl/Object.h>
#endif
#ifndef __DCL_EXCEPTION_H__
#include <dcl/Exception.h>
#endif
#ifndef __DCL_INPUT_STREAM_H__
#include <dcl/InputStream.h>
#endif
#ifndef __DCL_LISTED_HASH_MAP_H__
#include <dcl/ListedHashMap.h>
#endif

__DCL_BEGIN_NAMESPACE

class HttpServletContext;

/*
	HttpCookieDecoder
	HttpQueryStringDecoder
	HttpQueryStringEncoder

	InputStream
		HttpFormDataInputStream
	HttpFormData
		BufferedHttpFormData
		StoredHttpFormData
	HttpFormDataDecoder
*/

class DCLCAPI HttpCookieDecoder
{
public:
	static void decode(
			const ByteString& _contents,
			ListedStringToStringMap& _results
			);
};

class DCLCAPI HttpQueryStringDecoder
{
public:
	static void decode(
			const ByteString& _queryString,
			ListedStringToStringArrayMap& _results
			);

	static void decode(
		const char* _begin,
		const char* _end,
		ListedStringToStringArrayMap& _results
	);

	static bool isValidType(
			const char* _contentType
			);
};

class DCLCAPI HttpQueryStringEncoder
{
public:
	static String encode(
			const ListedStringToStringArrayMap& _map
			);
};

// RFC 1867 - Form-based File Upload in HTML
// http://www.faqs.org/rfcs/rfc1867.html

class HttpFormDataDecoder;
class DCLCAPI HttpFormData : public Object
{
	DECLARE_CLASSINFO(HttpFormData)
protected:
	struct PartHeader
	{
		// Content-Disposition
		String type;			// "form-data", "attachement"
		String name;			// value of "name"
		String filename;		// value of "filename"
		String contentType;		// value of "Content-Type"

		String toString() const;
	};

	// PartHeader::strFileName이 있을경우 파일의 시작이다.
	// 만약 onFileStart내부에서 에러가 발생하여 false를 리턴하면
	// 이후의 onFileData, onFileEnd는 호출되지 않는다.
	virtual bool onFileStart(
			const PartHeader& header,
			void**		ppCallbackData,
			String&		strCallbackError	// return false 이면
			);

	// formdata의 파일의 데이터가 있을때 불려진다.
	// 만약 onFileData 내부에서 에러가 발생하여 false를 리턴하면
	// onFileEnd 는 호출되지 않는다. 따라서 pCallbackData 관련하여
	// 자원이 할당되어 있으면 리턴하기 전에 해제해야 한다.
	virtual bool onFileData(
			const void* pData,
			size_t		nSize,
			void*		pCallbackData,
			String&		strCallbackError	// return false 이면
			);
	
	// 종결 part가 나타나면 불려진다. 종결 part는 CRLF로 구분되며
	// 만약 이것이 타나타지 않은 상태에는 bDataSuccess는 false를 넘긴다.
	virtual bool onFileEnd(
			const PartHeader& header,
			void*		pCallbackData,
			bool		bDataSuccess,
			String&		strCallbackError	// return false 이면
			);

	friend class HttpFormDataDecoder;
};

class DCLCAPI HttpFormDataDecoderException : public Exception
{
	DECLARE_CLASSINFO(HttpFormDataDecoderException)
public:
	enum ErrorCode
	{
		ePostReadError,
		eFormDataCallbackError
	};

	ErrorCode	m_errorCode;
	String		m_strMsg;

	HttpFormDataDecoderException(
		ErrorCode errorCode,
		IOException* pDetailEx
		);

	HttpFormDataDecoderException(
		ErrorCode errorCode,
		const String& _message
		);

	virtual String toString() const;
};


class DCLCAPI HttpFormDataDecoder : public Object
{
	DECLARE_CLASSINFO(HttpFormDataDecoder)
public:
	static bool isValidType(const char* _contentType);
	static ByteString getBoundary(const char* _contentType);

public:
	HttpFormDataDecoder(size_t	_bufferSize = 4096);

	virtual ~HttpFormDataDecoder();

	// 디코딩 과정중에 입력데이터와 관련한 에러가 발생하면
	// warnings() 을 통해 에러 메시지를 얻을 수 있다.
	// 에러가 발생한 데이터는 버려진다.
	void decode(
			InputStream&	_input,
			const char*		_contentType,
			size_t			_contentLength,
			ListedStringToStringArrayMap& _mapForm,
			HttpFormData&			_mapFormFile
			) __DCL_THROWS1(HttpFormDataDecoderException*);

	const String warnings() const { return __warnings; }

private:
	// false : 버퍼가 비어 있거나, not found CRLF
	bool getLine(char*& _begin, char*& _end);

	// true : valid first part boundary
	// false : close boundary, or invalid data
	bool getFirstBoundary(const ByteString& _boundary);

	// false EOF
	bool getPartHeader(HttpFormData::PartHeader& header)
			__DCL_THROWS1(HttpFormDataDecoderException*);

	// NULL : 버퍼가 비어 있거나 데이터가 유효하지 않다.
	enum DataState
	{
		dsNeedMoreData,
		dsBeforeNextBoundary,
		dsBeforeCloseBoundary
	};
		
	DataState getDataBlock(
			const ByteString& _boundary,
			char*& _begin,
			char*& _end
			);

	// false data empty && EOF
	bool readInput() __DCL_THROWS1(HttpFormDataDecoderException*);
	void appendWarning(const String& _warning);


	InputStream*	__input;
	size_t			__contentLength;
	size_t			__remainder;

	char*		__buffer;
	size_t		__bufferSize;
	char*		__begin;
	char*		__end;

	StringBuilder		__warnings;		// decoding warnings, string delimiter '\n'

	// const strings for compare
	_CONST ByteString	__contentDisposition;		// "Content-Dispositon"
	_CONST ByteString	__contentType;				// "Content-Type"
	_CONST ByteString	__name;						// "name"
	_CONST ByteString	__filename;					// "filename"
};

// 파일데이터를 버퍼에 유지
class DCLCAPI BufferedHttpFormData : public HttpFormData
{
	DECLARE_CLASSINFO(BufferedHttpFormData)
public:
	class FileInfoArray;

	class DCLCAPI FileInfo
	{
	public:
		String	filename;		// IE는 절대경로 포함, Netscape는 basename
		String	contentType;
		String	transferEncoding;
		ByteString fileData;

	protected:
		// not using
		//		FileInfo info = v[i];
		// use
		//		FileInfo& info = v[i];
		FileInfo();

		friend class FileInfoArray;
		friend class BufferedHttpFormData;
	};

	class DCLCAPI FileInfoArray
	{
	public:
		FileInfo& operator[] (size_t _index);
		size_t size() const;
		bool isEmpty() const;
		const String& name() const { return __name; }

	private:
		void*	__handle;

	protected:
		String	__name;		// <input name="name"

		// not using
		//		FileInfoArray v = formData["name"];
		// use
		//		FileInfoArray& v = formData.byName("name");
		//	or	FileInfoVecotr& v = formData[i];
		FileInfoArray(const String& _name);
		FileInfoArray(const FileInfo& src);
		~FileInfoArray();
		void add(FileInfo* pNewItem);

		friend class BufferedHttpFormData;
	};

public:
	BufferedHttpFormData();
	virtual ~BufferedHttpFormData();

	// <input name="name" type="file"/> 에서 "name" 이 개수
	// 서로 다른 "name"의 개수
	// FileInfoVector의 개수
	size_t size() const;
	bool isEmpty() const;
	FileInfoArray& operator[] (size_t _index);
	FileInfoArray& byName(const wchar_t* _name);

private:
	void*	__handle;

	void insert(const String& _name, FileInfo* pNewItem);

protected:
	virtual bool onFileStart(
			const PartHeader& header,
			void** ppCallbackData,
			String& strCallbackError	// return false 이면
			);

	virtual bool onFileData(
			const void* pData,
			size_t	nSize,
			void* pCallbackData,
			String& strCallbackError	// return false 이면
			);

	virtual bool onFileEnd(
			const PartHeader& header,
			void* pCallbackData,
			bool		bDataSuccess,
			String& strCallbackError	// return false 이면
			);
};

// 파일데이터를 임시파일에 저장
class DCLCAPI StoredHttpFormData : public HttpFormData
{
	DECLARE_CLASSINFO(StoredHttpFormData)
public:
	class FileInfoArray;

	class DCLCAPI FileInfo
	{
	public :
		String	filename;		// basename
		String	contentType;
		String	transferEncoding;
		size_t	fileSize;
		String	tempFilename;

	protected:
		// not using
		//		FileInfo info = v[i];
		// use
		//		FileInfo& info = v[i];
		FileInfo();
		FileInfo(const FileInfo& str);
		~FileInfo();

		friend class FileInfoArray;
		friend class StoredHttpFormData;
	};

	class DCLCAPI FileInfoArray
	{
	public:
		FileInfo& operator[] (size_t _index);
		size_t size() const;
		bool isEmpty() const;
		const String& name() const { return __name; }

	private:
		void*	__handle;

	protected:
		String	__name;		// <input name="name"

		// not using
		//		FileInfoArray v = formData["name"];
		// use
		//		FileInfoArray& v = formData.byName("name");
		//	or	FileInfoVecotr& v = formData[i];
		FileInfoArray(const String& _name);
		FileInfoArray(const FileInfo& src);
		~FileInfoArray();
		void add(FileInfo* pNewItem);

		friend class StoredHttpFormData;
	};

public:
	StoredHttpFormData(const String& strTempDir);
	virtual ~StoredHttpFormData();

	// <input name="name" type="file"/> 에서 "name" 이 개수
	// 서로 다른 "name"의 개수
	// FileInfoVector의 개수
	size_t size() const;
	bool isEmpty() const;
	FileInfoArray& operator[] (size_t _index);
	FileInfoArray& byName(const wchar_t* _name);

private:
	String	__tempDir;
	void*	__handle;

	void insert(const String& _name, FileInfo* pNewItem);

protected:
	virtual bool onFileStart(
			const PartHeader& header,
			void**		ppCallbackData,
			String&		strCallbackError	// return false 이면
			);

	virtual bool onFileData(
			const void* pData,
			size_t		nSize,
			void*		pCallbackData,
			String&		strCallbackError	// return false 이면
			);

	virtual bool onFileEnd(
			const PartHeader& header,
			void*		pCallbackData,
			bool		bDataSuccess,
			String&		strCallbackError	// return false 이면
			);
};

__DCL_END_NAMESPACE

#endif	// __DCL_HTTP_COLLECTION_H__
