#ifndef __DCL_POLL_THREAD_H__
#define __DCL_POLL_THREAD_H__		20110224

#ifndef __DCL_CONFIG_H__
#include <dcl/Config.h>
#endif

#ifndef __DCL_POLL_ABLE_H__
#include <dcl/PollAble.h>
#endif
#ifndef __DCL_THREAD_H__
#include <dcl/Thread.h>
#endif
#ifndef __DCL_LIST_T_H__
#include <dcl/ListT.h>
#endif

__DCL_BEGIN_NAMESPACE

/**
 * Non-Blocking 입출력 장치에대한 비동기 감시기능을 제공한다.
 * <p>UNIX에서는 poll을 사용하여 구현되어 있으며, 모든 입출력 fd에 대하여 사용하능하다.</p>
 * <p>Windows에서는 WSAPoll를 사용한다.
 * 이 때문에, Windows은 Windows Vista, Windows 2008 이상의 버전(_WIN32_WINNT >= 0x0600)
 * 의 WinSock2의 SOCKET을 감시에서만 사용할 수 있다.
 * Windows에서 다른 입출력에 대한 감시는 관련 클래스를 참고한다.</p>
 *
 * @see SerialPollThread
 * @see SocketPollThread
 *
 * @author	Daejung Kim
 * @since		DCL Version 3.0
 */
class DCLCAPI PollThread : public Thread
{
	DECLARE_CLASSINFO(IoPollThread)
public:
	PollThread(const char_t* _name = NULL);

	/**
	 * 입출력 감시 대상을 추가한다.
	 * @param _events
	 *		UNIX man page poll, Windows SDK WSAPoll을 참고하라.
	 * @return
	 *		스레드가 종료 중 이면 추가를 위한 큐삽입에 실패하고 false를 반환한다.
	 */
	virtual bool add(PollAble* _pPollAble, short _events)
		__DCL_THROWS1(IOException*);

	/**
	 * 입출력 감시대상을 제거한다.
	 * @return
	 *		스레드가 종료 중 이면 제거를 위한 큐삽입에 실패하고 false를 반환한다.
	 */
	virtual bool remove(PollAble* _pPollAble);

	/**
	 * 스레드를 종료 플래그를 설정한다.
	 * <p>이것을 호출 한 후에는 Thread::join를 사용하여
	 * 스레드 종료를 확인해야 한다.</p>
	 * <p>이것은 WSAPoll을 인터럽트 하지 않는다.
	 * 따라서, Windows에서 이것은 PollAble::onEvent 에서 사용하여야 즉시 효과가 있다.</p>
	 */
	void terminate();

protected:
	virtual int run();

	/**
	 * 입출력 감시대상에서 제거된 후 해당 객체를 처분한다.
	 * <p>기본구현은 _pPollAble의 destroy()를 호출하여 파괴한다<p>
	 */
	virtual void onRemoved(PollAble* _pPollAble);

	struct Update
	{
		PollAble*	pPollAble;
		bool		bAdd;

		Update() {
			pPollAble = NULL;
			bAdd = false;
		}

		Update(PollAble* _pPollAble, bool _bAdd)
		{
			this->pPollAble = _pPollAble;
			this->bAdd = _bAdd;
		}
	};
#ifdef _MSC_VER
#pragma warning (push)
#pragma warning (disable: 4251)
#endif
	typedef List<Update> UpdateList;
	UpdateList			__updateList;
	Thread::Mutex		__updateLock;
#ifdef _MSC_VER
#pragma warning (pop)
#endif

#if !__DCL_WINDOWS
	Thread::Event		__pollInterrupt;
#endif
	volatile bool		__terminate;
};

__DCL_END_NAMESPACE

#endif		// __DCL_POLL_THREAD_H__
