#ifndef __DCL_SQLCORE_H__
#define __DCL_SQLCORE_H__		20050322

#ifndef __DCL_OBJECT_H__
#include <dcl/Object.h>
#endif

#ifndef __DCL_LISTED_HASH_MAP_H__
#include <dcl/ListedHashMap.h>
#endif

__DCL_BEGIN_NAMESPACE

class DCLCAPI SQL
{
public:

enum Error
{
	eNoError = 0,		
	eServerError,			// 서버 혹은 서버 API 호출 에러이다.

	eBadAddress,			// 버퍼 혹은 파라메터의 주소가 잘못되었다.
	eOutOfMemory,			// 메모리 할당이 불가능 하다.
	eNotSupportMethod,		// 서버 혹은 드라이버에서 지원하지 않는 메소드 입니다.

	// Connection, Query
	eNotSupportStatement,

	// Connection Errors
	eInvalidConnectionString,	
	eConnected,				// Connection이 이미 연결되었다.
	eNotConnected,			// 먼저 Connection::open을 호출해야 한다.
	eInTransaction,			// 이미 트랜잭션이 시작되었다.
	eNotInTransaction,		// 트랜잭션 상태가 아니다.
	eHaveChildQuery,		// 이 연결을 사용하는 Query가 있다.
	eNoChildQuery,			// 파괴하려는 Query는 다른곳에서 생성되었다.
	eNotAvailable,			// 값을 얻을 수 없다. for geServerInfo

	// Query Errors
	eNotPrepared,			// 먼저 Query::prepare 를 호출해야 한다.
	eNotExecuted,			// 먼저 Query::execute 를 호출해야 한다.
	eNotFetched,			// 먼저 Query::fetch 를 호출해야 한다.
	eInvalidIndex,			// Field나 Param을 참조하는 잘못된 Index값이다.

	// Field
	eValueIsNull,			// 값이 null 이다
	eNotSupportDataType,	// 서버 혹은 드라이버에서 지원하지 않는 데이터타입이다.
	eInvalidDataType,		// 필드값을 얻거나 파라메터 값을 설정하기 위한 데이터타입이 잘못되었다.
	eInvalidBufferSize,		// 필드 혹은 아웃바인드의 값을 읽고자 하는 버퍼의 크기가 유효하지 않다.
							// Field::getValue *_pn에 필요한 버퍼의 크기를 리턴
	eOutOfRange,			// typeNumeric에서 버퍼의 데이터타입과 사이즈에 적용하는데 범위를 초과했다.
	
	// Param
	eInvalidData,			// Param::setData, 잘못된 데이터이다.
							// 서버API로의 데이터 변환에 실패하였다.
	eInvalidDataSize		// 데이터타입에 해당하는 데이터의 사이즈가 유효하지 않다.
							// for SQLParam::setValue
};

enum DataType
{
	typeUnknown	= 0,		// C-DataType			SQL-Type
	typeInteger,			// int32_t, int64_t		TINYINT, SMALLINT, INTEGER, BIGINT
	typeUInteger,			// uint32_t, uint64_t	BIT, UTINYINT, USMALLINT, UINTEGER, UBIGINT
	typeFloat,				// float, double		REAL, DOUBLE, FLOAT
	typeNumeric,			// INTx, UINTx, float, double, char*	NUMBER, DECIMAL
	typeDate,				// SQL::Date			DATE
	typeTime,				// SQL::Time			TIME
	typeTimeTz,				// SQL::Time			TIME WITH TIME ZONE
	typeTimeStamp,			// SQL::TimeStamp		TIMESTAMP
	typeTimeStampTz,		// SQL::TimeStamp		TIMESTAMP WITH TIME ZONE
	typeInterval,			// SQL::Interval
	typeIntervalYm,			// SQL::Interval		INTERVAL YEAR TO SECOND
	typeIntervalDs,			// SQL::Interval		INTERVAL DAY TO SECOND
	typeText,				// char*				CHAR, VARCHAR,
	typeBinary,				// byte_t*				BINARY, RAW
	typeLongText,			// char*				LONG
	typeLongBinary,			// byte_t*				LONG RAW
	typeClob,				// Clob					direct accessible CLOB
	typeBlob,				// Blob					direct accessible BLOB

	// buffer type
	typeInputStream,		// InputStream
	typeOutputStream		// OutputStream
};

static bool getErrorMessage(
			wchar_t* _buf,		// OUT
			size_t*	_buflen,	// IN, OUT
			Error _errorCode	// IN
			);
static const wchar_t* dataTypeName(DataType _dataType);

struct Date
{
	int16_t		year;		// 1 ~ 9999, -4712 ~ 9999, DBMS dependent
	uint8_t		month;		// 1 ~ 12
	uint8_t		day;		// 1 ~ 31
};

struct Time
{
	uint8_t		hour;		// 0 ~ 23
	uint8_t		min;		// 0 ~ 59
	uint8_t		sec;		// 0 ~ 59
	uint32_t	frac;		// 0 ~ 999999999, 1/1,000,000,000 fractional seconds
	int16_t		tzoff;		// -720~840, GMT-12 ~ GMT+14, Time offset, Minutes
};

struct TimeStamp
{
	int16_t		year;		// 1 ~ 9999, -4712 ~ 9999, DBMS dependent
	uint8_t		month;		// 1 ~ 12
	uint8_t		day;		// 1 ~ 31
	uint8_t		hour;		// 0 ~ 23
	uint8_t		min;		// 0 ~ 59
	uint8_t		sec;		// 0 ~ 59
	uint32_t	frac;		// 0 ~ 999999999, 1/1,000,000,000 fractional seconds
	int16_t		tzoff;		// -720~840, GMT-12 ~ GMT+14, Time offset, Minutes
};

// INTERVAL YEAR TO MONTH : nYears ~ nMonths
// INTERVAL DAY TO SECOND : nDays ~ nFracs
struct Interval
{
	int32_t		years;		// -999999999 ~ 999999999
	int8_t		months;		// -11 ~ 11
	int32_t		days;		// -999999999 ~ 999999999
	int8_t		hours;		// -23 ~ 23
	int8_t		mins;		// -59 ~ 59
	int8_t		secs;		// -59 ~ 59
	int32_t		fracs;		// -999999999 ~ 999999999, 1/1,000,000,000 fractional seconds
};

class Field;
class Param;
class Query;
class Connection;

// interface classes
class DCLCAPI Field : public Object
{
	DECLARE_CLASSINFO(SQL::Field)
public:
	const String& name() const;
	DataType dataType() const;
	short precision() const;
	short scale() const;
	Connection* connection() const;

	bool getDataSize(
			size_t*		_pn,		// count of bytes
			bool		_maxSize
			);

	bool getData(
			void*		_pv,		// buffer, OUT
			size_t*		_pn,		// sizeof *_pv, IN, OUT
			DataType	_dataType	// typeof *_pv
			);

	// interfaces
public:
	virtual bool isNull() const = 0;
	virtual const wchar_t* serverDataTypeName() const = 0;

protected:
	Field(Query* _queryHandle);
	virtual ~Field();

	virtual bool __getDataSize(
			size_t*		_pn,
			bool		_maxSize
			) = 0;

	virtual bool __getData(
			void*		_pv,
			size_t*		_pn,		// IN, OUT
			DataType	_dataType
			) = 0;

protected:
	Query* __queryHandle;
	String		__name;
	DataType	__dataType;
	short		__precision;
	short		__scale;
};

class DCLCAPI Param : public Field
{
	DECLARE_CLASSINFO(SQL::Param)
public:
	void setName(String _name);

	bool setData(
			_CONST		void* _pv,	// data, IN
			size_t		_n,			// sizeof *_pv
			DataType	_dataType,	// typeof *_pv
			DataType	_assignType	// assign to server type
			);

	// out param support
	bool setDataType(
			DataType	_dataType
			);

	// interfaces
	virtual void setNull() = 0;

protected:
	Param(Query* _queryHandle);
	virtual ~Param();

	virtual bool __setData(
			_CONST void* _pv,
			size_t		_n,
			DataType	_dataType,
			DataType	_assignType
			) = 0;

	// out param support
	virtual bool isNull() const;

	virtual bool __setDataType(
			DataType	_dataType
			);

	virtual bool __getDataSize(
			size_t*		_pn,
			bool		_maxSize
			);

	virtual bool __getData(
			void*		_pv,
			size_t*		_pn,		// IN, OUT
			DataType	_dataType
			);
};

class DCLCAPI Query : public Object
{
	DECLARE_CLASSINFO(SQL::Query)
public:
	bool prepare(const char* _sql, size_t _sqllen,
			size_t _paramCount);
	bool execute();
	bool fetch();
	bool nextResult();
	bool getField(
			size_t	_index,				// zero based index
			Field** _fieldHandleOut		// OUT
			);

	bool getParam(
			size_t	_index,				// zero based index
			Param** _paramHandleOut		// OUT
			);

	// properties
	bool eof() const;
	int64_t affectedRows() const;
	size_t fieldCount() const;
	size_t paramCount() const;
	wchar_t placeholder() const;

	Connection* connection() const;
	bool inState(unsigned int _state) const;

	// interfaces
protected:
	Query(Connection* _connHandle);
	virtual ~Query();

	virtual void __destroy() = 0;
	virtual bool __prepare(const char* _sql, size_t _sqllen,
			size_t _paramCount) = 0;
	virtual bool __execute() = 0;
	virtual bool __fetch() = 0;
	virtual bool __nextResult();

	virtual bool __getField(size_t _index, Field** _fieldHandleOut) = 0;
	virtual bool __getParam(size_t _index, Param** _paramHandleOut) = 0;

protected:
	Connection* __connHandle;
	bool		__eof;
	int64_t		__affectedRows;	// for INSERT, UPDATE

	size_t		__fieldCount;
	size_t		__paramCount;
	wchar_t		__placeholder;	// Oracle(':') or '?'

	friend class Connection;

	// Object States
protected:
	unsigned int	__states;

public:
	enum State
	{
		stStandBy	= 0x0001,
		stPrepared	= 0x0002,
		stExecuted	= 0x0004,
		stFetched	= 0x0008
	};
};

class DCLCAPI Connection : public Object
{
	DECLARE_CLASSINFO(SQL::Connection)
private:
	long		__refCount;

public:
	long refCount() const	{ return __refCount; }
	long addRef();		// new incremented __refCount
	long release();		// new decremented __refCount

public:
	bool open(const char* _connString, size_t _connlen);	
	bool close();
	bool execute(const char* _sql, size_t _sqllen);

	bool startTrans();
	bool commitTrans();
	bool rollbackTrans();	

	bool getErrorMessage(
			char* _pbuf, 
			size_t* _pn				// IN, OUT
			);

	bool getServerInfo(
			char* _pbuf, 
			size_t* _pn				// IN, OUT
			);

	bool createQueryInstance(Query** _queryHandleOut);
	bool destroyQueryInstance(Query* _queryHandle);

	// properties
	Error errorCode() const;
#if __DCL_DEBUG
	const wchar_t* errorFileName() const;
	int errorLine() const;
#endif
	bool canTransact() const;
	const wchar_t* serverTitle() const;
	bool inState(unsigned int uState) const;

	// interfaces
	virtual void destroy() = 0;
protected:
	Connection(const wchar_t* _serverTitle);
	virtual ~Connection();

	virtual bool __open(const char* _connString, size_t _connlen) = 0;	
	virtual bool __close() = 0;
	virtual bool __execute(const char* _sql, size_t _sqllen) = 0;

	virtual bool __startTrans() = 0;
	virtual bool __commitTrans() = 0;
	virtual bool __rollbackTrans() = 0;	

	virtual bool __createQueryInstance(Query** _queryHandleOut) = 0;
	virtual bool __getErrorMessage(char* _buf, size_t* _buflen) = 0;
	virtual bool __getServerInfo(char* _buf, size_t* _buflen) = 0;

protected:
	bool			__canTransact;
	const wchar_t*	__serverTitle;

	Error			__errorCode;
	const wchar_t*	__errorFileName;
	int				__errorLine;

	// Object States
	unsigned int	__states;
public:
	enum State
	{
		stClosed	= 0x0001,
		stOpenned	= 0x0002,
		stInTransaction = 0x0004
	};

__PROTECTED:
	// for SQLQuery, SQLField, SQLParam, and inherited classes
	void setErrorStatus(Error _errorCode, const wchar_t* _filename, int _line);

	/*
	pszConnectionString을 파싱하여 map에 저장한다.
	pszConnectionString의 예
		"USER NAME=aUser;PASSWORD=aPW;DATABASE NAME=aDB; SERVER NAME=aHost;"

	map의 KEY는 대문자로 변환되며, KEY와 VALUE 모두 space는 trim된다.

	RETUNR VALUE : map에 설정된 아이템이 개수
	*/
	static size_t splitConnectionString(
					const char* _connString, size_t _strlen,
					ListedByteStringToByteStringMap& _map
					);
};

struct DRIVER_MODULE
{
	// DCL common members
	uint32_t		uSize;				// structure size
	uint32_t		uDCLVersion;		// DCL_VERSION
	const wchar_t*	pszBuildTimeStamp;	// __TIMESTAMP__
	uint32_t		uBuildFlag;			// release(0), debug(1)
	uint32_t		uModuleType;		// DCL_SQL_DRIVER_MODULE
	const wchar_t*	pszDescription;		// module description

	// private members
	uint32_t		uVersion;			// DCL_SQL_VERSION
	const wchar_t*	pszServerTitle;
	const wchar_t*	pszFileVersion;
	bool			(*pfnInitialize)();
	bool			(*pfnCleanup)();
	Connection*		(*pfnCreateConnectionInstance)();
};

};	// class SQL

//#define __SET_ERROR(_errorCode) setErrorStatus(_errorCode, __THIS_FILE__, __LINE__)

#define __SET_STATE(state)		__states |= state
#define __UNSET_STATE(state)	__states &= ~state

#include <dcl/SQLCore.inl>

__DCL_END_NAMESPACE

#endif	// __DCL_SQLCORE_H__
