#ifndef __DCL_SERIAL_PORT_H__
#define __DCL_SERIAL_PORT_H__		20110223

#ifndef __DCL_CONFIG_H__
#include <dcl/Config.h>
#endif

#if __DCL_WINDOWS
	#define B1200		CBR_1200
	#define B2400		CBR_2400
	#define B4800		CBR_4800
	#define B9600		CBR_9600
	#define B19200		CBR_19200
	#define B38400		CBR_38400
	#define B57600		CBR_57600
	#define B115200		CBR_115200
#else
	#include <termios.h>
#endif

#ifndef __DCL_POLL_ABLE_H__
#include <dcl/PollAble.h>
#endif

__DCL_BEGIN_NAMESPACE

#if __DCL_WINDOWS
	class SerialPollThread;
#endif

/**
 * 직렬포트 관련 장치파일을 구성하고 접근한다.
 * <p>직렬포트에 대한 파일명은 UNIX에서  /dev/tty#N, /dev/ttyS#N, /dev/ttyUSB#N 과 같은 것들이고,
 * Windows에서는 COM#N 등 이다.</p>
 *<p>Linux에서 직렬포트는 프로세스가 uucp그룹에 포함되어 있지 않으면 접근이 거부되므로 확인한다.</p>
 *
 * <p>이 클래스는 직렬포트의 Non-Blocking 입출력에 대하여 준비되어 있다.
 * 자세한 사용방법은 {@link SerialPollThread}를 참고한다.</p>
 *
 * <p>USB-Serial 장치를 사용하는 경우 실행시간에 포트가 제거될 수 있다.
 * UNIX의 경우 {@link #onEvent(Events,SerialPollThread*)
 * onEvent(Events _events, SerialPollThread* _pWatcher)}의 _events에 POLLHUP가 전달 되고,
 * Windows의 경우 {@link File#read(void*,size_t) File::read(void* _buf, size_t _n)},
 * {@link File#write(constvoid*,size_t) File::write(const void* _buf, size_t n)}의 호출 과정에서
 * 예외를 처리해야 한다. Windows 에러코드는 ERROR_ACCESS_DENIED (5)이다.
 * </p>
 *
 * @see PollAble
 * @see SerialPollThread
 */
class DCLCAPI SerialPort : public PollAble
{
	DECLARE_CLASSINFO(SerialPort)
public:
	virtual ~SerialPort();

	/**
	 * 주어진 경로의 포트를 열고 객체를 구성한다.
	 * @see	#open(constString,int,size_t,size_t) open(const String& _path, int _oflags, size_t _inQueue, size_t _outQueue)}
	 */
	SerialPort(const String& _path, int _oflags,
		size_t _inQueue = 4096, size_t _outQueue = 4096
		) __DCL_THROWS1(IOException*);

	/**
	 * 주어진 경로의 포트를 연다.
	 * <p>파일을 열때 플래그는 File::READONLY, File::WRITEONLY,
	 * File::READWRITE과 File::NONBLOCK가 조합될 수 있고,
	 * 내부에서 File::NOCTTY가 적용된다.</p>
	 * <p>_inQueue, _outQueue는 Windows에만 사용하고 그 외에는 무시된다.
	 * 자세한 내용은 SetupComm 을 참고한다.<p>
	 */
	void open(const String&  _path, int _oflags,
		size_t _inQueue = 4096, size_t _outQueue = 4096
		) __DCL_THROWS1(IOException*);

	enum BaudRate
	{
		BR_1200		= B1200,
		BR_2400		= B2400,
		BR_4800		= B4800,
		BR_9600		= B9600,
		BR_19200		= B19200,
		BR_38400		= B38400,
		BR_57600		= B57600,
		BR_115200	= B115200
	};

	enum ByteSize
	{
		BS_8NO,			// 8Bit, No parity
		BS_7EVEN,		// 7Bit, Even parity
		BS_7ODD			// 7Bit, Odd parity
	};

	enum StopBits
	{
		SB_1,			// 1 Stop bits
		SB_2			// 2 Stop bits
	};

	enum FlowControl
	{
		FC_NONE,
		FC_RTSCTS,		// Hardware
		FC_XONXOFF		// Software
	};

	/**
	 * <p>포트의 속성을 설정한다.</p>
	 * <pre>
	 *	SerialPort port(String(_T("/dev/ttyS0"), false);
	 *	port.setAttributes(SerialPort::BR_9600, SerialPort::BS_8NO, SerialPort::SB_1, SerialPort::FC_NONE);
	 *	    ...
	 * </pre>
	 */
	void setAttributes(
		BaudRate _baudRate = BR_38400, ByteSize _byteSize = BS_8NO,
		StopBits _stopBits = SB_1, FlowControl _flowControl = FC_NONE
		) __DCL_THROWS1(IOException*);

	enum PurgeFlag
	{
		PURGE_RX = 0x0001,
		PURGE_TX = 0x0002
	};

	/**
	 * 전송되지 않거나, 또는 읽혀지지 않는 데이터를 버린다.
	 */
	void purge(int _flags) __DCL_THROWS1(IOException*);

protected:
	/**
	 * {@link SerialPollThread}에의해 감시되다가 이벤트가 발생되었을 때 호출된다.
	 * 
	 */
	virtual bool onEvent(short _revents, PollThread* _pPollThread)
		__DCL_THROWS1(IOException*);

#if __DCL_WINDOWS
private:
	// SerialPollThread에 의해 사용된다.
	DWORD			__revents;
	OVERLAPPED		__overlapped;

	bool waitCommEvent();

	friend SerialPollThread;
#endif

public:
	/**
	 * 주어진 경로의 가능 여부를 확인한다.
	 * <p>UNIX에서는 access(R_OK | W_OK)로 확인하고,
	 * Windows에서는 경로를 열어보고 그 결과를 반환한다.</p>
	 */
	static bool access(const String& _path);
};

#if __DCL_WINDOWS
	#undef B1200
	#undef B2400
	#undef B4800
	#undef B9600
	#undef B19200
	#undef B38400
	#undef B57600
	#undef B115200
#endif

__DCL_END_NAMESPACE

#endif		// __DCL_SERIAL_PORT_H__
