#ifndef __DCL_SOCKET_H__
#define __DCL_SOCKET_H__		20030501

#ifndef __DCL_CONFIG_H__
#include <dcl/Config.h>
#endif

#if __DCL_WINDOWS
	#ifndef _WINSOCK2API_
		#error "Required winsock2.h, See dcl/_windows.h"
	#endif
	#include <ws2tcpip.h>
	#ifdef _MSC_VER
		#pragma comment(lib, "ws2_32.lib")
	#endif
#else
	#include <sys/socket.h>
	#include <sys/un.h>
	#include <netinet/in.h>
	#include <arpa/inet.h>
#endif

#ifndef __DCL_POLL_ABLE_H__
	#include <dcl/PollAble.h>
#endif

__DCL_BEGIN_NAMESPACE

#if __DCL_WINDOWS
	typedef int socklen_t;
	class SocketPollThread;
#endif

/**
 * 소켓에 대한 파일 인터페이스와 Non-Blocking 입출력을 제공한다.
 *
 * @see PollThread
 * @see SocketPollThread
 * @see man tcp
 */
class DCLCAPI Socket : public PollAble
{
	DECLARE_CLASSINFO(Socket)
public:
	// Object overrides
	virtual String toString() const;

	// File overrides
	void open(const String& _addr, uint16_t _port)
			__DCL_THROWS1(IOException*);

	virtual size_t available() const
			__DCL_THROWS1(IOException*);

	virtual size_t read(void* _buf, size_t _n)
			__DCL_THROWS1(IOException*);

	virtual size_t write(const void* _buf, size_t _n)
			__DCL_THROWS1(IOException*);

	struct DCLCAPI Addr
	{
		union
		{
			short					sa_family;
			struct sockaddr_in		sa_in;		// AF_INET
			struct sockaddr_in6		sa_in6;		// AF_INET6
#if !__DCL_WINDOWS
			struct sockaddr_un		sa_un;		// AF_UNIX
#endif
		};

		Addr();

		/**
		 * AF_INET, AF_INET6 주소를 만든다.
		 * <p>IPv4, IPv6 주소값 검사를 하고 이를 실패하면 호스트명 으로부터 주소를 만든다.
		 * _WIN32_WINNT < 0x0600 인 경우 IPv4만 사용된다.
		 * </p>
		 */
		Addr(const char* _addr, uint16_t _port)
			__DCL_THROWS1(IOException*);

		Addr(const String& _addr, uint16_t _port)
			__DCL_THROWS1(IOException*);

#if !__DCL_WINDOWS
		/**
		 * AF_UNIX 주소를 만든다.
		 */
		Addr(const char* _path)
			__DCL_THROWS1(IOException*);

		Addr(const String& _path)
			__DCL_THROWS1(IOException*);

#endif
		String toString() const;	
	};

	/**
	 * Listen 소켓 객체에 주소를 설정한다.
	 *
	 * <p>bind()호출 전에 create()를 호출하지 않았으면,
	 * _my_addr.sa_family, _type, _protocol을 사용하여 소켓을 생성한다.
	 *
	 * @param	_reuse
	 *		setsockopt를 사용하여 리슨소켓을 재사용하도록 설정한다.
	 */
	void bind(const Addr& _my_addr,
		int _type = SOCK_STREAM, int _protocol = 0,	bool _reuse = true
		) __DCL_THROWS1(IOException*);

	void accept(Socket& _r)
		__DCL_THROWS1(IOException*);

	void connect(const Addr& _serv_addr)
		__DCL_THROWS1(IOException*);

	socklen_t getsockname(Addr& _addr)
		__DCL_THROWS1(IOException*);

	socklen_t getpeername(Addr& _addr)
		__DCL_THROWS1(IOException*);

	// API implements

	Socket();

	Socket(const String& _addr, uint16_t _port)
			__DCL_THROWS1(IOException*);

	virtual ~Socket();

		
	virtual void close()
		__DCL_THROWS1(IOException*);

	void create(int _domain = AF_INET, int _type = SOCK_STREAM, int _protocol = 0)
		__DCL_THROWS1(IOException*);

	void setNonblock()
		__DCL_THROWS1(IOException*);

	void bind(const struct sockaddr* _my_addr, socklen_t _addrlen)
		__DCL_THROWS1(IOException*);

	void listen(unsigned _backlog = 5)
		__DCL_THROWS1(IOException*);

	void accept(Socket& r, struct sockaddr* _addr, socklen_t* _addrlen)
		__DCL_THROWS1(IOException*);

	virtual void connect(const sockaddr* _serv_addr, socklen_t _addrlen)
		__DCL_THROWS1(IOException*);

	/**
	 * @return Non-Blocking 상태에서 0을 반환할 수 있다. 이 경우 다시 시도해야 한다.
	 */
	virtual size_t send(const void* _buf, size_t _n, int _flags = 0)
		__DCL_THROWS1(IOException*);

	/**
	 * @return Non-Blocking 상태에서 0을 반환할 수 있다. 이 경우 다시 시도해야 한다.
	 */
	virtual size_t recv(void* _buf, size_t _n, int _flags = 0)
		__DCL_THROWS1(IOException*);
	
protected:
	/**
	 * Blocking IO를 하는 소켓 인스턴스를 위해 포함한다.
	 * <p>만약 Non-Blocking IO를 위해서는 이것을 override하여야 한다.</p>
	 */
	virtual bool onEvent(short _revents, PollThread* _pPollThread)
		__DCL_THROWS1(IOException*);

#if __DCL_WINDOWS
private:
	// SocketPollThread에 의해 사용된다.
	WSAEVENT		__waitEvent;

	friend class SocketPollThread;
#endif
};

__DCL_END_NAMESPACE

#endif		// __DCL_SOCKET_H__
