#ifndef __DCL_STRING_WRITER_H__
#define __DCL_STRING_WRITER_H__		20110221

#ifndef __DCL_WRITER_H__
#include <dcl/Writer.h>
#endif
#ifndef __DCL_STRING_H__
#include <dcl/String.h>
#endif

__DCL_BEGIN_NAMESPACE

/**
 * wchar_t 스트림을 버퍼링하여 String을 만든다.
 *
 * @author	Daejung Kim
 * @since		DCL Version 3.0
 *
 * @see		CharBuffer
 * @see		String
 * @see		Writer
 */
class DCLCAPI StringWriter : public Writer
{
	DECLARE_CLASSINFO(StringWriter)
public:
	/**
	 * 버퍼링된 데이터로 String 객체를 구성하여 반환다.
	 */
	virtual String toString() const;

	/**
	 * 주어진 값으로 객체를 구성한다.
	 * <p>
	 * {@link #write(constwchar_t*,size_t) write(const wchar_t* _buf, size_t _n)}이
	 * 호출되기 전에는 내부버퍼를 생성하지 않는다.
	 * </p>
	 * @param _capacity
	 *		내부버퍼가 생성될 때 wchar_t 문자의 개수
	 */
	StringWriter(size_t _capacity = 4096);

	/**
	 * 객체의 파괴를 수행한다.
	 * close하지 않았으면 {@link #close()}하며 예외가 발생하면 무시한다.
	 */
	virtual ~StringWriter();

	/**
	 * 버퍼를 삭제하고 스트림을 닫는다.
	 */
	virtual void close()
			__DCL_THROWS1(IOException*);

	/**
	 * 텍스트를 쓴다
	 * @param _buf
	 *		버퍼
	 * @param _n
	 *		버퍼의 크기가 _n개의 wchar_t 문자 
	 * @return
	 *		Writer&
	 */
	virtual Writer& write(const wchar_t* _buf, size_t _n)
			__DCL_THROWS1(IOException*);

	/**
	 * 형식화된 문자열을 출력한다.
	 * @param _format
	 *		포멧 문자열. %s는 %ls로 해석된다.
	 * @param _arglist
	 *		아귀먼트 리스트
	 * @return
	 *		출력된 총 wchar_t 문자의 개수를 반환한다.
	 */
	virtual int vprintf(const wchar_t* _format, va_list _arglist)
		__DCL_THROWS1(IOException*);

	/**
	 * 주어진 Writer로 내부버퍼의 텍스트를 쓴다. 내부버퍼는 그대로 유지한다.
	 *
	 * @param _writer
	 *		A Writer
	 * @return
	 *		_writer에 쓴 wchar_t 문자의 개수
	 */
	virtual size_t writeTo(Writer& _writer) const
			__DCL_THROWS1(IOException*);

	/**
	 * 주어진 Writer로 내부버퍼의 텍스트를 내보내고, 내부버퍼를 reset한다.
	 *
	 * @param _writer
	 *		A Writer
	 * @return
	 *		_writer에 쓴 wchar_t 문자의 개수
	 */
	virtual size_t flushTo(Writer& _writer)
			__DCL_THROWS1(IOException*);

	/**
	 * 내부버퍼를 초기화 한다.
	 */
	virtual void reset();

	/**
	 * 내부버퍼의 시작 위치를 반환하다.
	 * 만약, 단 한번도 {@link #write(constwchar_t*,size_t) write(const wchar_t* _buf, size_t _n)}이 호출되지 않았으면,
	 * NULL을 반환할 것이다.
	 */
	const wchar_t* data() const;

	/**
	 * 내부버퍼에 저장되어 있는 문자열의 길이를 반환한다. 
	 */
	size_t size() const;

protected:
	CharBuffer*		__buf;
	size_t			__capacity;
};

inline const wchar_t* StringWriter::data() const
{
	return __buf != NULL ? __buf->data() : NULL;
}

inline size_t StringWriter::size() const
{
	return __buf != NULL ? __buf->__dataLength : 0;
}

__DCL_END_NAMESPACE

#endif		// __DCL_STRING_WRITER_H__
