#ifndef __DCL_THREAD_H__
#define __DCL_THREAD_H__	20110109

#ifndef __DCL_CONFIG_H__
#include <dcl/Config.h>
#endif

#if __DCL_WINDOWS
	#ifndef _WINDOWS_
		#error "Required windows.h, See dcl/_windows.h"
	#endif
	#if __DCL_PTHREAD
		#include <pthread.h>
	#endif
#else
	#include <bits/pthreadtypes.h>
	#define INFINITE	-1
#endif

#ifndef __DCL_OBJECT_H__
#include <dcl/Object.h>
#endif
#ifndef __DCL_STRING_H__
#include <dcl/String.h>
#endif

__DCL_BEGIN_NAMESPACE

/**
 * 운영체제 스레드 및 관련 API를 추상화한다.
 */
class SysError;
class DCLCAPI Thread : public Object
{
	DECLARE_CLASSINFO(Thread)
public:
	/**
	 * 객체의 간략한 정보를 반환한다.
	 */
	virtual String toString() const;

	/**
	 * 기본 생성자이다.
	 * @param _name
	 *			객체에 이름을 준다.
	 */
	Thread(const char_t* _name = NULL);

	/**
	 * 스레드를 생성하고 시작한다.
	 * 실패하면 예외를 던진다.
	 */
	 void start() __DCL_THROWS1(SysError*);

	/**
	 * 스레드의 종료를 기다린다.
	 * @return
	 *		{@link #run()}이 반환한 값
	 */
	int join();

	/**
	 * 스레드의 시작여부를 반환한다.
	 * @return
	 *		{@link #start()} 호출로 스레드가 시작되었으면 true, 그렇지 않으면 false를 반환한다.
	 */
	bool started() const { return __threadId != 0; }
	
	/**
	 * 스레드 ID를 반환한다.
	 */
	unsigned long id() const { return __threadId; }

	/**
	 * 스레드 생성에 주어진 이름을 반환한다.
	 */
	const String& name() const { return __name; }

protected:
	/**
	 * 스레드 객체를 초기화한다.
	 * <p>이 메소드는 {@link #start()}에 의해 생성된 스레드 내에서 호출된다.</p>
	 * <p>기본 구현은 {@link #getCurrentThread()}을 위해
	 * 객체의 포인터를 스레드 전용 기억장소에 설정한다.<p>
	 * @see
	 *		#keySetValue
	 * @return
	 *		false를 반환하면 스레드는 즉시 종료되고,
	 *		{@link #run()}은 호출되지않고 {@link #join()}은 -1을 반환한다.
	 */
	virtual bool init();

	/**
	 * 스레드의 주 실행 루프를 구현한다.
	 */
	virtual int run() = 0;

private:
	unsigned long	__threadId;
	String				__name;

#if __DCL_PTHREAD
	static void* startRoutine(void* _pThread);
#elif __DCL_WINDOWS
	HANDLE			__hThread;
	static DWORD WINAPI startRoutine(void* _pThread);
#endif

public:
	/**
	 * 미리초만큼 스레드의 실행을 중지한다. UNIX 에서 시그널에 인터럽트될 수 있다.
	 */
	static void sleep(unsigned int _mills); 

	/**
	 * CPU를 다른 스레드에 양보한다.
	 */
	static void yield();

	/**
	 * 호출한 스레드의 스레드의 ID을 얻는다.
	 */
	static unsigned long getCurrentThreadId();

	/**
	 * 호출한 스레드의 객체를 반환한다.
	 * <p>이 메소드는 스레드 생성 후 {@link #init()}에서 스레드 전용 기억장소에 설정한 것의 반환값이다.
	 * 따라서, {@link #init()}이 호출되지 않으면 NULL을 리턴한다.
	 */
	static Thread* getCurrentThread();

	/**
	 * 다중 스레드 환경에서 단일 호출을 위한 제어변스의 타입을 정의한다.
	 */
#if __DCL_PTHREAD
	typedef pthread_once_t	OnceType;
	enum { ONCE_INIT = 0 /* PTHREAD_ONCE_INIT */ };
#elif __DCL_WINDOWS
	typedef long					OnceType;
	enum { ONCE_INIT = (long)0 };
#endif
	typedef void (* initRoutine)();
	/**
	 * 다중 스레드 환경에서 _initRoutine의 단일 호출을 보장한다.
	 * <p>사용의 예는 다음과 같다.</p>
	 * <pre>
	 *    static Thread::OnceType once = Thread::ONCE_INIT;
	 *    static void initRoutine()
	 *    {
	 *	    ...
	 *    }
	 *    
	 *    Thread::once(once, initRoutine);
	 */
	static void once(OnceType& _onceControl, initRoutine _initRoutine);

#if __DCL_PTHREAD
	typedef pthread_key_t	KeyType;
#elif __DCL_WINDOWS
	typedef DWORD			KeyType;
#endif

	/**
	 * 스레드 전용 지억장소를 위한 키를 생성한다.
	 * <p>UNIX는 pthread_key_create, Windows는 TlsAlloc을 사용한다.</p>
	 * @throws SysError*
	 *		키 생성이 실패하면 예외가 던져지며, 사유는 다음과 같다.
	 * <ul>
	 *    <li>UNIX - 키생성에 PTHREAD_KEYS_MAX에 도달했거나 메모리가 부족한 경우이다.</li>
	 *    <li>Windows - TlsAlloc이 TLS_OUT_OF_INDEXES을 반환했다.</li>
	 * </ul>
	 */
	static KeyType keyCreate() __DCL_THROWS1(SysError*);

	/**
	 * 스레드 전용 기억장소를 위한 키를 삭제한다
	 *
	 * @param _key
	 *		{@link #keyCreate}로 반환된 값.
	 * @throws SysError*
	 *		_key가 유효하지 않으면 예외가 던져진다.
	 */
	static void keyDelete(KeyType _key) __DCL_THROWS1(SysError*);

	/**
	 * 호출 스레드 전용 기억장소에 값을 설정한다.
	 *
	 * @param _key
	 *		{@link #keyCreate}로 반환된 값.
	 * @throws SysError*
	 *		_key가 유효하지 않으면 예외가 던져진다.
	 */
	static void keySetValue(KeyType _key, void* _value) __DCL_THROWS1(SysError*);

	/**
	 * 호출 스레드 전용 기억장소의 값을 리턴한다.
	 *
	 * @param _key
	 *		{@link #keyCreate}로 반환된 값.
	 * @return
	 *		값이 설정되어 있지 않으면 NULL을 반환한다.
	 * @throws SysError*
	 *		_key가 유효하지 않으면 예외가 던져진다.
	 */
	static void* keyGetValue(KeyType _key) __DCL_THROWS1(SysError*);

	/**
	 * C 런타임이 다중 스레드에 안전하지 않으면 이것을 사용하여 동기화 한다.
	 * <p>crtLock을 호출하면 반드시 이 후에 crtUnlock을 호출 해야만 한다.<p>
	 * <p>example</p>
	 * <pre>
	 * char a[100];
	 * time_t now = time(NULL);
	 * Thread::crtLock(&now);
	 * char* cp = ctime(&now);
	 * strcpy(a, cp);
	 * Thread::crtUnlock(&now);
	 * </pre>
	 */
	static void crtLock(const void* _p);
	static void crtUnlock(const void* _p);

	static long incrementAndGet(volatile long& _n);
	static long long incrementAndGet(volatile long long& _n);
	static long decrementAndGet(volatile long& _n);
	static long long decrementAndGet(volatile long long& _n);

public:
	class DCLCAPI Event
	{
	private:
#if __DCL_PTHREAD
		enum { READ_FD_INDEX = 0 };
		int			__fds[2];
#elif __DCL_WINDOWS
		HANDLE	__hEvent;
#endif

	__protected:
#if __DCL_PTHREAD
		int handle() { return __fds[READ_FD_INDEX]; }
#elif __DCL_WINDOWS
		HANDLE handle() { return __hEvent; }
#endif

	public:
		Event();
		~Event();
		void set();
		void reset();

		/**
		 @return false는 timed-out
		 Windows: GetLastError() == ERROR_TIMEOUT
		 UNIX: errno == ETIMEDOUT, EINTR (signal에 인터럽트 될 수 있다)
		 */
		bool wait(unsigned int _milliseconds = INFINITE);
		bool isWaiting()	{ return __bWaiting; }
	private:
		volatile bool __bWaiting;
	};

	class DCLCAPI Mutex
	{
	private:
#if __DCL_PTHREAD
		pthread_mutex_t		__mutex;
#elif __DCL_WINDOWS
		CRITICAL_SECTION	__cs;
#endif

#if __DCL_WINDOWS
	protected:
		Mutex(DWORD _dwSpinCount);
#endif
	public:
		Mutex();
		~Mutex();
		void lock();
		bool tryLock();
		void unlock();
	};

#if __DCL_PTHREAD
	class DCLCAPI SpinLock
	{
	private:
		pthread_spinlock_t	__spinLock;

	public:
		SpinLock(bool _pshared = false);
		~SpinLock();
		void lock();
		bool tryLock();
		void unlock();
	};
#elif __DCL_WINDOWS
	class DCLCAPI SpinLock : public Mutex
	{
	public:
		SpinLock() : Mutex(INFINITE) { }
	};
#endif


#if __DCL_PTHREAD || (__DCL_WINDOWS && _WIN32_WINNT >= 0x0600)
	class DCLCAPI ReadWriteLock
	{
	private:
#if __DCL_PTHREAD
		pthread_rwlock_t	__rwlock;
#else
		SRWLOCK				__srwlock;
#endif

	public:
		ReadWriteLock();
		~ReadWriteLock();
		void readLock();			// WINNT AcquireSRWLockShared
		void writeLock();			// WINNT AcquireSRWLockExclusive
#if __DCL_PTHREAD
		bool tryReadLock();
		bool tryWriteLock();
		void unlock();
#else
		void readUnlock();		// WINNT ReleaseSRWLockShared
		void writeUnlock();		// WINNT ReleaseSRWLockExclusive
#endif
	};
#endif

	/**
	 *	CondVar: Conditional Variable
	 *		Windows Note. Major Version >= 6, Vista, 2008
	 *		See. InitializeCondVaritionVariable
	 */
	class DCLCAPI CondVar
	{
	private:
#if __DCL_PTHREAD
		pthread_cond_t	__cond;
#elif __DCL_WINDOWS
	#if _WIN32_WINNT >= 0x0600
		CONDITION_VARIABLE	__cond;
	#else
		Event				__cond;
	#endif
#endif

	public:
		CondVar();
		~CondVar();
		void signal();
#if __DCL_PTHREAD || (__DCL_WINDOWS && _WIN32_WINNT >= 0x0600)
		void broadcast();
#endif
		
		/**
		@return false는 timed-out
		Windows: GetLastError() == ERROR_TIMEOUT
		UNIX: errno == ETIMEDOUT
		*/
		bool wait(Mutex& _mutex, unsigned int _milliseconds = INFINITE);
#if __DCL_WINDOWS && _WIN32_WINNT >= 0x0600
		bool wait(ReadWriteLock& _lock, bool shared, unsigned int _milliseconds = INFINITE);
#endif
		bool isWaiting()	{ return __bWaiting; }
	private:
		volatile bool		__bWaiting;
	};

#if __DCL_PTHREAD
	class DCLCAPI Barrier
	{
	private:
		pthread_barrier_t	__barrier;

	public:
		Barrier(unsigned int _count);
		~Barrier();
		void wait();
	};
#endif

	template<typename TYPE>
	class SingleLock
	{
	private:
		TYPE&		__lock;		// Mutex or SpinLock

	public:
		SingleLock(TYPE& _lock) : __lock(_lock)
		{
			__lock.lock();
		}

		~SingleLock()
		{
			__lock.unlock();
		}
	};

	typedef SingleLock<Mutex>		SingleLockMutex;
	typedef SingleLock<SpinLock>	SingleLockSpin;

	template<typename TYPE>
	class SingleTryLock
	{
	private:
		TYPE&		__lock;			// Mutex or SpinLock
		bool	__locked;
	public:
		SingleTryLock(TYPE& _lock) : __lock(_lock)
		{
			__locked = __lock.tryLock();
		}
		~SingleTryLock()
		{
			if (__locked)
				__lock.unlock();
		}
	};

	typedef SingleTryLock<Mutex>		SingleTryLockMutex;
	typedef SingleTryLock<SpinLock>	SingleTryLockSpin;

};


#if __DCL_WINDOWS
inline long Thread::incrementAndGet(volatile long& _n)
{
	return InterlockedIncrement(&_n);
}

inline long Thread::decrementAndGet(volatile long& _n)
{
	return InterlockedDecrement(&_n);
}

#if _WIN32_WINNT >= 0x0600
inline long long Thread::decrementAndGet(volatile long long& _n)
{
	return InterlockedDecrement64(&_n);
}

inline long long Thread::incrementAndGet(volatile long long& _n)
{
	return InterlockedIncrement64(&_n);
}
#endif
#endif

__DCL_END_NAMESPACE

#endif	// __DCL_THREAD_H__
