#ifndef __DCL_URI_H__
#define __DCL_URI_H__

#ifndef __DCL_EXCEPTION_H__
#include <dcl/Exception.h>
#endif

__DCL_BEGIN_NAMESPACE

/*
	References
		http://www.faqs.org/rfcs/rfc2396.html
		RFC 2396 
			- Uniform Resource Identifiers (URI)
				: Generic Syntax
*/

class DCLCAPI URLDecoder
{
public:
	static ByteString decode(const char* _begin, const char* _end);
	static ByteString decode(const ByteString& _str);
	static String decode(const String& _str);
};

class DCLCAPI URLEncoder
{
public:
	static ByteString encode(const ByteString& _str);
	static String encode(const String& _str);
};

class URISyntaxException;

class DCLCAPI URI : public Object
{
	DECLARE_CLASSINFO(URI)
public:
	URI(const wchar_t* _uri)
		__DCL_THROWS1(URISyntaxException*);

	URI(const wchar_t* _scheme,
		const wchar_t* _schemeSpecificPart,
		const wchar_t* _fragment
		) __DCL_THROWS1(URISyntaxException*);

	URI(const wchar_t* _scheme,
		const wchar_t* _userInfo,
		const wchar_t* _host,
		int _port,
		const wchar_t* _path,
		const wchar_t* _query,
		const wchar_t* _fragment
		) __DCL_THROWS1(URISyntaxException*);

	URI(const wchar_t* _scheme,
		const wchar_t* _host,
		const wchar_t* _path,
		const wchar_t* _fragment
		) __DCL_THROWS1(URISyntaxException*);

	URI(const wchar_t* _scheme,
		const wchar_t* _authority,
		const wchar_t* _path,
		const wchar_t* _query,
		const wchar_t* _fragment
		) __DCL_THROWS1(URISyntaxException*);

	const String& scheme() const;
	const String& schemeSpecificPart() const;
	const String& authority() const;
	const String& userInfo() const;
	const String& host() const;
	int port() const;
	const String& path() const;
	const String& query() const;
	const String& fragment() const;

	String toString() const;

	bool isAbsolute() const;

protected:
	String	__scheme;
	String	__schemeSpecificPart;
	String	__authority;
	String	__userInfo;
	String	__host;
	int		__port;
	String	__path;
	String	__query;
	String	__fragment;

	String	__uri;

	void parse(const wchar_t* _uri)
		__DCL_THROWS1(URISyntaxException*);

	void assign(
		const wchar_t* _scheme,
		const wchar_t* _userInfo,
		const wchar_t* _host,
		int _port,
		const wchar_t* _path,
		const wchar_t* _query,
		const wchar_t* _fragment
		) __DCL_THROWS1(URISyntaxException*);
};

class DCLCAPI URISyntaxException : public Exception
{
	DECLARE_CLASSINFO(URISyntaxException)
public:
	URISyntaxException(const String& _reason, const String& _source);
	virtual String toString() const;

protected:
	String	__message;
};

inline const String& URI::scheme() const
{
	return __scheme;
}

inline const String& URI::schemeSpecificPart() const
{
	return __schemeSpecificPart;
}

inline const String& URI::authority() const
{
	return __authority;
}

inline const String& URI::userInfo() const
{
	return __userInfo;
}

inline const String& URI::host() const
{
	return __host;
}

inline int URI::port() const
{
	return __port;
}

inline const String& URI::path() const
{
	return __path;
}

inline const String& URI::query() const
{
	return __query;
}

inline const String& URI::fragment() const
{
	return __fragment;
}

inline String URI::toString() const
{
	return __uri;
}

inline bool URI::isAbsolute() const
{
	return !__scheme.isEmpty();
}

__DCL_END_NAMESPACE

#endif	// __DCL_URI_H__
