#ifndef __DCL_XML_PARSER_H__
#define __DCL_XML_PARSER_H__			20110703

#ifndef __DCL_CONFIG_H__
#include <dcl/Config.h>
#endif

#ifdef _MSC_VER
	#ifdef _DLL
		#if defined(_DEBUG)
			#pragma comment(lib, "expatd.lib")
		#else
			#pragma comment(lib, "expat.lib")
		#endif
	#else
		#define XML_STATIC
		#ifdef _DEBUG
			#pragma comment(lib, "libexpatd.lib")
		#else
			#pragma comment(lib, "libexpat.lib")
		#endif
	#endif
#endif

#include <expat.h>

#ifndef __DCL_OBJECT_H__
#include <dcl/Object.h>
#endif
#ifndef __DCL_EXCEPTION_H__
#include <dcl/Exception.h>
#endif
#ifndef __DCL_ARRAY_H__
#include <dcl/Array.h>		// StringStringArray
#endif
#ifndef __DCL_WRITER_H__
#include <dcl/Writer.h>
#endif

__DCL_BEGIN_NAMESPACE

class XmlParser;

class DCLCAPI XmlException : public Exception
{
	DECLARE_CLASSINFO(XmlException)

protected:
	String	__message;

public:
	XmlException(XML_Parser _parser);
	virtual String toString() const;
};

class DCLCAPI XmlDefaultHandler
{
	DECLARE_CLASSINFO(XmlDefaultHandler)

protected:
	static void startElement(void* _userData, const XML_Char* _name, const XML_Char** _attrs);
	static void endElement(void* _userData, const XML_Char* _name);
	static void characters(void* _userData, const XML_Char* _chars, int _len);

	virtual void startElement(const String& _name, const StringStringArray& _attrs);
	virtual void endElement(const String& _name);
	virtual void characters(const String& _chars);

	friend class XmlParser;
};

/**
 * XML을 파싱한다.
 * <p>내부적으로 Expat XML Parser가 사용된다.
 * </p>
 *
 *	@see http://expat.sourceforge.net/
 *	@see XmlDefaultHandler
 *	@see XmlElement
 *	@see XmlDocument
 */
class DCLCAPI XmlParser : public Object
{
	DECLARE_CLASSINFO(XmlParser)

protected:
	XML_Parser	__parser;

public:
	XmlParser();
	virtual ~XmlParser();

	/**
	 * XML 파싱을 수행한다.
	 *
	 * @param _data
	 *		UTF-8 엔코딩 바이트 문자열
	 * @param _handler
	 *		XmlDefaultHandler
	 */
	void parse(const ByteString& _data, XmlDefaultHandler& _handler)
		__DCL_THROWS1(XmlException*);

};

class DCLCAPI XmlNode : public Object
{
	DECLARE_CLASSINFO(XmlNode)

protected:
	XmlNode*		__pParentNode;

public:
	XmlNode(XmlNode* _pParentNode) { __pParentNode = _pParentNode; }
	XmlNode* parentNode() const { return __pParentNode; }
};

class DCLCAPI XmlCharsNode : public XmlNode
{
	DECLARE_CLASSINFO(XmlCharsNode)

protected:
	String				__chars;

public:
	virtual String toString() const;

	XmlCharsNode(XmlNode* _pParentNode, const String& _chars);
};

class DCLCAPI XmlElement : public XmlNode
{
	DECLARE_CLASSINFO(XmlElement)

protected:
	String				__name;			// element name

	StringStringArray	__attrs;		// attributes
	PointerArray		__children;		// array of XmlNode

public:
	virtual String toString() const;

	XmlElement(XmlNode* _pParentNode,
		const String& _name, const StringStringArray& _attrs);
	virtual ~XmlElement();

	void addChildNode(XmlNode* pNode);

	const String& name() const { return __name; }
	const StringStringArray& attrs() const { return __attrs; }
	const PointerArray& children() const { return __children; }
	const String innerText() const;
	const String innerXml() const;
};

class DCLCAPI XmlDocument : public XmlDefaultHandler
{
	DECLARE_CLASSINFO(XmlDocument)

protected:
	virtual void startElement(const String& _name, const StringStringArray& _attrs);
	virtual void endElement(const String& _name);
	virtual void characters(const String& _chars);

	XmlElement*		__pCurrentElement;
	XmlElement*		__pTopElement;

public:
	virtual String toString() const;

	XmlDocument();
	virtual ~XmlDocument();

	const XmlElement* topElement() const { return __pTopElement; }
};

DCLCAPI inline Writer& operator << (Writer& _writer, const XmlNode& _obj)
	__DCL_THROWS1(IOException*)
{
	_writer << _obj.toString();
	return _writer;
}

DCLCAPI inline Writer& operator << (Writer& _writer, const XmlDocument& _obj)
	__DCL_THROWS1(IOException*)
{
	_writer << _obj.toString();
	return _writer;
}


__DCL_END_NAMESPACE

#endif			// __DCL_XML_PARSER_H__
